/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.builder;

import org.mule.runtime.ast.api.ArtifactAst;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.function.UnaryOperator;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Allows to set a mapping function that will resolve properties in the parameters of the owning {@link ArtifactAst}.
 * <p>
 * The mapping function must transform a {@link String} containing keys surrounded with placeholder delimiters (<code>${}</code>)
 * to a {@link String} with those placeholders replaced with the appropriate property.
 *
 * @since 1.0
 */
public class PropertiesResolver implements UnaryOperator<String> {

  private static final Logger LOGGER = LoggerFactory.getLogger(PropertiesResolver.class);

  private final List<Runnable> mappingFunctionChangedCallbacks = new ArrayList<>();

  private UnaryOperator<String> mappingFunction = UnaryOperator.<String>identity();

  @Override
  public String apply(String propertyPlaceHolder) {
    if (propertyPlaceHolder == null) {
      return null;
    }

    LOGGER.trace("Resolving property: '{}'...", propertyPlaceHolder);
    final String value = mappingFunction.apply(propertyPlaceHolder);

    if (Objects.equals(propertyPlaceHolder, value)) {
      LOGGER.trace("Did not resolve property: '{}'", propertyPlaceHolder);
    } else {
      LOGGER.debug("Resolved property: '{}' to '{}'", propertyPlaceHolder, value);
    }

    return value;
  }

  public void setMappingFunction(UnaryOperator<String> mappingFunction) {
    LOGGER.debug("setMappingFunction: '{}'...", mappingFunction);
    this.mappingFunction = mappingFunction;
    mappingFunctionChangedCallbacks.forEach(Runnable::run);
  }

  public void onMappingFunctionChanged(Runnable callback) {
    mappingFunctionChangedCallbacks.add(callback);
  }
}
