/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.cfg.api;

import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.ERROR_HANDLER;
import static org.mule.runtime.api.message.error.matcher.ErrorTypeMatcherUtils.createErrorTypeMatcher;
import static org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.DEFAULT_GROUP_NAME;
import static org.mule.sdk.api.stereotype.MuleStereotypes.FLOW;
import static org.mule.sdk.api.stereotype.MuleStereotypes.SUB_FLOW;
import static java.lang.String.format;
import static java.util.Collections.reverse;
import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static java.util.Optional.empty;
import static java.util.Optional.of;


import org.mule.runtime.api.meta.model.error.ThrowsErrors;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.cfg.internal.node.errorhandling.ErrorHandlingContext;
import org.mule.runtime.cfg.internal.node.ChainedExecutionPathNodeBuilder;
import org.mule.runtime.cfg.internal.node.ReferencedChainNode;
import org.mule.runtime.cfg.internal.node.RouterExecutionPathNodeBuilder;
import org.mule.runtime.cfg.internal.node.ScopeExecutionPathNodeBuilder;
import org.mule.runtime.cfg.internal.node.SimpleOperationNode;
import org.mule.runtime.cfg.internal.node.SourceNode;
import org.mule.runtime.cfg.internal.node.errorhandling.ErrorHandlingExecutionPathNodeBuilder;
import org.mule.runtime.cfg.internal.node.errorhandling.ErrorHandlerNode;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * Factory that creates the {@link ChainExecutionPathTree} corresponding to the given {@link ComponentAst}.
 * 
 * @since 1.1.
 */
public class ChainExecutionPathTreeFactory {

  // TODO (W-12591953) Handle this using information from the extension model
  private final static String FLOW_REF = "flow-ref";
  private final static String RAISE_ERROR = "raise-error";

  private static final String ANY_POSSIBLE_ERROR = "MULE:ANY";

  private final ArtifactAst application;
  private final Map<String, ChainExecutionPathTree> cachedTrees = new HashMap<>();

  public ChainExecutionPathTreeFactory(ArtifactAst application) {
    this.application = application;
  }

  /**
   * Creates an {@link ChainExecutionPathTree} for the given {@link ComponentAst} representing a chain (for instance, a
   * {@link #FLOW}
   *
   * @param chainComponentAst the chain {@link ComponentAst} to generate the execution path tree for.
   * @return an execution path tree representing how the chain would be executed.
   */
  public ChainExecutionPathTree generateFor(ComponentAst chainComponentAst) {
    return cachedTrees
        .computeIfAbsent(chainComponentAst.getLocation().getLocation(),
                         location -> recursiveGenerateFor(chainComponentAst, application, new ErrorHandlingContext()));
  }

  private ChainExecutionPathTree recursiveGenerateFor(ComponentAst chainComponentAst, ArtifactAst artifactAst,
                                                      ErrorHandlingContext errorHandlers) {
    switch (chainComponentAst.getComponentType()) {
      case OPERATION:
        return getOperationNode(chainComponentAst, errorHandlers);
      case SOURCE:
        return new SourceNode(chainComponentAst);
      case FLOW:
      case CHAIN:
      case ROUTE:
        return getNestedChainFrom(chainComponentAst, artifactAst, errorHandlers);
      case ROUTER:
        RouterExecutionPathNodeBuilder router = new RouterExecutionPathNodeBuilder(chainComponentAst);
        chainComponentAst.directChildren()
            .forEach(routeChild -> router
                .withRoute(recursiveGenerateFor(routeChild, artifactAst,
                                                propagatedErrorHandlerContext(chainComponentAst, errorHandlers))));
        return router.withErrorHandlerContext(errorHandlers).build();
      case SCOPE:
        return new ScopeExecutionPathNodeBuilder(chainComponentAst)
            .withChild(getNestedChainFrom(chainComponentAst, artifactAst,
                                          propagatedErrorHandlerContext(chainComponentAst, errorHandlers)))
            .withErrorHandlerContext(errorHandlers)
            .build();
      case ERROR_HANDLER:
        // We need to treat them in reverse order so the first one to appear later on is the first error handler and not the last
        // one
        // TODO (W-12450002) - Error handlers of same level shouldn't be linked
        chainComponentAst.directChildren().stream().collect(collectingAndThen(toList(), l -> {
          reverse(l);
          return l;
        })).forEach(child -> recursiveGenerateFor(child, artifactAst, errorHandlers));
        return null;
      case ON_ERROR:
        processErrorHandler(chainComponentAst, artifactAst, errorHandlers);
        return null;
      default:
        throw new IllegalStateException(format("ComponentType of ComponentAST is not currently handled by the factory. Component: %s. Location: %s",
                                               chainComponentAst.getIdentifier().getName(),
                                               chainComponentAst.getLocation().getLocation()));
    }
  }

  private void processErrorHandler(ComponentAst chainComponentAst, ArtifactAst artifactAst,
                                   ErrorHandlingContext errorHandlers) {
    String errorExpression = chainComponentAst.getParameter(DEFAULT_GROUP_NAME, "type").getRawValue();
    if (errorExpression == null) {
      errorExpression = ANY_POSSIBLE_ERROR;
    }
    ErrorHandlerNode errHandler = new ErrorHandlingExecutionPathNodeBuilder(chainComponentAst)
        .setChild(getNestedChainFrom(chainComponentAst, artifactAst, errorHandlers))
        .setErrorMatcher(createErrorTypeMatcher(artifactAst.getErrorTypeRepository(), errorExpression)).build();
    if (chainComponentAst.getIdentifier().getName().equals("on-error-propagate")) {
      errorHandlers.addPropagateHandler(errHandler);
    } else if (chainComponentAst.getIdentifier().getName().equals("on-error-continue")) {
      errorHandlers.addContinueHandler(errHandler);
    } else {
      // TODO Think about on-error and other error-handling references (W-12392496)
      throw new IllegalStateException("Identifier type not supported: " + chainComponentAst.getIdentifier().getName());
    }
  }

  private ChainExecutionPathTree getNestedChainFrom(ComponentAst chainComponentAst, ArtifactAst artifactAst,
                                                    ErrorHandlingContext errorHandlers) {
    List<ComponentAst> children = new ArrayList<>(chainComponentAst.directChildren());
    // Error handler should be processed first, so it is available while processing the rest of the chain
    Optional<ComponentAst> hasEH = getErrorHandler(children);
    hasEH.ifPresent(err -> recursiveGenerateFor(err, artifactAst, errorHandlers));
    ChainedExecutionPathNodeBuilder builder = new ChainedExecutionPathNodeBuilder(chainComponentAst);
    children.forEach(child -> builder.addChild(recursiveGenerateFor(child, artifactAst, errorHandlers)));
    // The error handler has now to be removed because in following chains, nodes, etc... this error handler doesn't apply anymore
    hasEH.ifPresent(err -> errorHandlers.removeLastHandlers(err.directChildren().size()));
    ChainExecutionPathTree node = builder.build();
    if (node == null) {
      throw new RuntimeException("empty route/scope/flow");
    }
    return node;
  }

  private Optional<ComponentAst> getErrorHandler(List<ComponentAst> children) {
    // If there is an error handler, it would be the last component of the chain. We also remove it in that case from the rest of
    // the elements of the chain, to avoid re-processing it (which would be an error, since it's not part of the actual execution
    // chain, and it could also lead to references from this error handler to itself)
    if (children.size() == 0) {
      return empty();
    }
    return children.get(children.size() - 1).getComponentType().equals(ERROR_HANDLER) ? of(children.remove(children.size() - 1))
        : empty();
  }

  private Optional<ComponentAst> getTopLevelElementWithName(String name) {
    return application.topLevelComponentsStream().filter(ast -> ast.getComponentId().map(id -> id.equals(name)).orElse(false))
        .findFirst();
  }

  private boolean propagatesErrorHandlingContext(ComponentAst componentAst) {
    return componentAst.getModel(ThrowsErrors.class).map(model -> model.getErrorModels().isEmpty()).orElse(true);
  }

  private ErrorHandlingContext propagatedErrorHandlerContext(ComponentAst ast, ErrorHandlingContext context) {
    return propagatesErrorHandlingContext(ast) ? context : new ErrorHandlingContext(false);
  }

  private ChainExecutionPathTree getOperationNode(ComponentAst chainComponentAst, ErrorHandlingContext errorHandlers) {
    if (chainComponentAst.getIdentifier().getName().equals(RAISE_ERROR)) {
      // TODO (W-12390971)
    } else if (chainComponentAst.getIdentifier().getName().equals(FLOW_REF)) {
      Optional<ComponentParameterAst> parameter = chainComponentAst.getParameters().stream()
          .filter(param -> param.getModel().getAllowedStereotypes().stream()
              .anyMatch(stereotypeModel -> stereotypeModel.isAssignableTo(FLOW) || stereotypeModel.isAssignableTo(SUB_FLOW)))
          .findFirst();
      Optional<ComponentAst> flowComponent = parameter.flatMap(param -> getTopLevelElementWithName(param.getRawValue()));
      if (flowComponent.isPresent()) {
        // if it is not present, it could be because it is a dynamic reference, in which case we don't have a full flow
        // reference
        return new ReferencedChainNode(chainComponentAst, new LazyValue<>(() -> generateFor(flowComponent.get())));
      }
    }
    return new SimpleOperationNode(chainComponentAst, errorHandlers);
  }

}
