/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.cfg.api;

import org.mule.runtime.api.message.Error;
import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.ast.api.ComponentAst;

/**
 * Provides a way to navigate a {@link ChainExecutionPathTree} with its
 * {@link ChainExecutionPathTree#accept(ChainExecutionPathTreeVisitor)} method not coupled to its internal structure.
 * <p>
 * For each item of interest in the {@link ChainExecutionPathTree} being navigated, a method from this visitor will be called. The
 * order of the called methods is consistent and depends on the definition of the chain. It is up to implementations to keep the
 * proper context for backtracking after an inner chain or an error handler is finished.
 * <p>
 * Implementations need not be tread-safe.
 *
 * @since 1.1
 */
public interface ChainExecutionPathTreeVisitor {

  /**
   * Called when a {@link #SOURCE} is found in the execution tree.
   * <p>
   * If the chain is a {@link #FLOW} with a {@link #SOURCE}, this will be the first method called when visiting the chain.
   *
   * @param source the tree representing visited source
   */
  void visitSource(ChainExecutionPathTree source);

  /**
   * Called when a simple {@link #OPERATION} is found in the execution tree.
   * <p>
   * A simple {@link #OPERATION} is any operation within a chain that is not a {@link #SCOPE} or {@link #ROUTER}.
   *
   * @param operation the visited simple operation.
   * @return {@code true} if the visitor should traverse through the error handlers related to this operation.
   */
  void visitSimpleOperation(ChainExecutionPathTree operation);

  /**
   * Called when a {@link #SCOPE} is found in the execution tree.
   * <p>
   * After this method is called, if it returns {@code true} it will proceed to visit its inner components. After that,
   * {@link #scopeFinished(ChainExecutionPathTree)} will be called and then the method for the component right after the scope
   * will be called.
   *
   * @param scope the visited scope.
   * @return {@code true} if the visitor should visit the inner components of the {@link #SCOPE}, or {@code false} if it should
   *         just skip to the next element. If it is {@code true}, then every inner component of this particular {@link #SCOPE}
   *         will be visited.
   */
  boolean visitScope(ChainExecutionPathTree scope);

  /**
   * Called after the {@link #ROUTE}/{@link #CHAIN} within a {@link #SCOPE} was visited.
   * <p>
   * This is a delimiter call. This will only be called after {@link #visitScope(ChainExecutionPathTree)} and will be referred to
   * the last call to that method. If {@link #visitRouter(ChainExecutionPathTree)} returned {@code false}, this method won't be
   * called.
   */
  void scopeFinished(ChainExecutionPathTree scope);

  /**
   * Called when a {@link #ROUTER} is found in the execution tree.
   * <p>
   * After this method is called, if it returns {@link true}, for each of its inner chains
   * {@link #innerRouteStarted(ChainExecutionPathTree)} (ComponentAst)} followed by the calls that correspond to its inner
   * components, and then {@link #routerFinished(ChainExecutionPathTree)} will be called. After all inner chains have been
   * navigated, {@link #routerFinished(ChainExecutionPathTree)} will be called with the same {@link ComponentAst} of this
   * {@link #ROUTER}, and the method for the component right after the scope will be called.
   *
   * @param router the visited router.
   * @return {@code true} if the visitor should visit the inner components of the {@link #ROUTER}, or {@code false} if it should
   *         just skip to the next element.
   */
  boolean visitRouter(ChainExecutionPathTree router);

  /**
   * Called when a {@link #ROUTE} within a {@link #ROUTER} is found in the execution tree.
   * <p>
   * This is a delimiter call. The finish of the innerChain started here will be notified by a call to
   * {@link #innerRouteFinished(ChainExecutionPathTree)}. This wil only be called if {@link #visitRouter(ChainExecutionPathTree)}
   * returned true.
   *
   * @param innerChain the visited innerChain
   */
  void innerRouteStarted(ChainExecutionPathTree innerChain);

  /**
   * Called after the last component in a {@link #ROUTE} within a {@link #ROUTER}.
   * <p>
   * This is a delimiter call. This will only be called after {@link #innerRouteStarted(ChainExecutionPathTree)} and will apply to
   * the innerChain indicated in the last call to that method.
   *
   * @param innerChain the visited innerChain
   */
  void innerRouteFinished(ChainExecutionPathTree innerChain);

  /**
   * Called after the last {@link #ROUTE} within a {@link #ROUTER} was visited.
   * <p>
   * This is a delimiter call. This will only be called after {@link #visitRouter(ChainExecutionPathTree)} and will be referred to
   * the last call to that method. If {@link #visitScope(ChainExecutionPathTree)} returned {@code false}, this method won't be
   * called.
   */
  void routerFinished(ChainExecutionPathTree router);

  /**
   * This method will be called right after {@link #visitSimpleOperation(ChainExecutionPathTree)},
   * {@link #visitScope(ChainExecutionPathTree)} or {@link #visitRouter(ChainExecutionPathTree)} if any of the following apply:
   * <ul>
   * <li>There is an {@link #ERROR_HANDLER} for an {@link Error} raised by the last visited processor applicable to it.</li>
   * <li>There is an {@link #ERROR_HANDLER} for the ANY {@link ErrorType} applicable for the last visited processor.</li>
   * </ul>
   * <p>
   * For every possible error that the {@param component} may throw, if there is an error handler that manages that error, it will
   * be invoked (following the propagation rules).
   * <p>
   * This is a delimiter call. The finish of the error handling started here will be notified by a call to
   * {@link #errorHandlingFinishedFor(ChainExecutionPathTree)}.
   * 
   * @param component that raised the error.
   * @return {@code true} if the error handling for this {@param component} should be traversed
   *         ({@link #errorHandlingFinishedFor(ChainExecutionPathTree)} won't be invoked if this returns {@code false}).
   */
  boolean errorHandlingStartsFor(ChainExecutionPathTree component);

  /**
   * This method will be called before starting the traversal of an error handler. This will be called after a
   * {@link #errorHandlerStarted(ErrorType, ChainExecutionPathTree)} was called.
   * <p>
   * For a particular {@param errorType} thrown by the component that executed
   * {@link #errorHandlingStartsFor(ChainExecutionPathTree)}, this method will be executed for the direct error handler that
   * handles the error, and if there is the need to continue the propagation (i.e. this is an on error propagate handler), it will
   * also invoke the next one that corresponds, in the propagation hierarchy.
   * <p>
   * This is a delimiter call. The finish of the error handler started here will be notified by a call to
   * {@link #errorHandlerFinished(ChainExecutionPathTree)}.
   *
   * @param errorType the type of the error handled by the visited error handler
   * @param handler   the visited error handler
   * @return {@code true} if the visitor should travers this error handler ({@link #errorHandlerFinished(ChainExecutionPathTree)}
   *         won't be invoked if this returns {@code false}).
   */
  boolean errorHandlerStarted(ErrorType errorType, ChainExecutionPathTree handler);

  /**
   * Called after the last component in an {@link #ERROR_HANDLER}.
   * <p>
   * This is a delimiter call. This will only be called after {@link #errorHandlerStarted(ErrorType, ChainExecutionPathTree)}.
   *
   * @param handler the visited error handler.
   */
  void errorHandlerFinished(ChainExecutionPathTree handler);

  /**
   * Called after the last component in an {@link #ERROR_HANDLER} if the error has finished to be propagated to other error
   * handlers. This is, for example, if the current {@param handler} is an on-error-continue, or if we reached the last level of
   * error handling to be able to handle the error (e.g. the main flow). This method will be called for every {@param errorType}
   * that the originating component may throw, as far as there is at least an error handler to handle it.
   * <p>
   * This is a delimiter call. The finish of the error handler started here will be notified by a call to
   * {@link #errorHandlerFinished(ChainExecutionPathTree)}, including for this last {@param handler}, to then call this method.
   *
   * @param errorType the type of the error handled by the visited error handler
   * @param handler   the visited error handler
   */
  void errorHandlerPropagationComplete(ErrorType errorType, ChainExecutionPathTree handler);

  /**
   * Called after the last {@link #errorHandlerPropagationComplete(ErrorType, ChainExecutionPathTree)} (for the last error) for
   * the given {@param component}.
   * <p>
   * This is a delimiter call. This will only be called after {@link #errorHandlingStartsFor(ChainExecutionPathTree)} (called with
   * the same {@param component}.
   *
   * @param component the {@link ComponentAst} that originates the errors that involved the execution of error handlers.
   */
  void errorHandlingFinishedFor(ChainExecutionPathTree component);

  /**
   * Called after the last component of the chain unless {@link #chainFinishedSuccessWithResponse(ChainExecutionPathTree)} is
   * called.
   */
  void chainFinishedSuccess();

  /**
   * Called after the last component of the chain when the chain is a {@link #FLOW} with a {@link #SOURCE} that sends a response.
   * <p>
   * If this method is called, {@link #chainFinishedSuccess()} is not.
   *
   * @param source the source of a flow that sends a response
   */
  void chainFinishedSuccessWithResponse(ChainExecutionPathTree source);

  /**
   * Called after the last component of the chain within an error handler that propagates the error unless
   * {@link #chainFinishedErrorWithResponse(ChainExecutionPathTree)} is called.
   */
  void chainFinishedError();

  /**
   * Called after the last component of the chain within an error handler that propagates the error when the chain is a
   * {@link #FLOW} with a {@link #SOURCE} that sends a response.
   * <p>
   * If this method is called, {@link #chainFinishedError()} is not.
   *
   * @param source the source of a flow that sends a response
   */
  void chainFinishedErrorWithResponse(ChainExecutionPathTree source);

}
