/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.cfg.internal.node;

import static java.util.Optional.empty;
import static java.util.Optional.of;

import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.cfg.api.ChainExecutionPathTree;
import org.mule.runtime.cfg.api.ChainExecutionPathTreeVisitor;

import java.util.Optional;
import java.util.function.Predicate;


public class ChainedExecutionPathNodeBuilder {

  private final ComponentAst chainAst;
  private ChainNode first;
  private ChainNode last;
  private boolean continueAdding = true;

  public ChainedExecutionPathNodeBuilder(ComponentAst chainAst) {
    this.chainAst = chainAst;
  }

  public ChainedExecutionPathNodeBuilder addChild(ChainExecutionPathTree child) {
    if (!continueAdding) {
      return this;
    }
    ChainNode newNode = new ChainNode(child);
    if (first == null) {
      first = newNode;
    } else {
      last.next = of(newNode);
    }
    last = newNode;

    // WE ARE NOT CONSIDERING IF THE ERROR IS HANDLED WITHIN THIS CONTEXT! (TODO W-12390971)
    continueAdding =
        !child.allExecutionPathsContain(ast -> ast.getComponentAst().getIdentifier().getName().equals("raise-error"));
    return this;
  }

  public ChainExecutionPathTree build() {
    return new HeadNode(chainAst, first != null ? first : new ChainNode(new NullNode()));
  }

  public static class HeadNode implements ChainExecutionPathTree {

    private final ComponentAst chainAst;
    private final ChainNode firstNode;

    public HeadNode(ComponentAst chainAst, ChainNode firstNode) {
      this.chainAst = chainAst;
      this.firstNode = firstNode;
    }

    @Override
    public void accept(ChainExecutionPathTreeVisitor visitor) {
      this.firstNode.accept(visitor);
    }

    @Override
    public ComponentAst getComponentAst() {
      return this.chainAst;
    }

    @Override
    public boolean anyExecutionPathContains(Predicate<ChainExecutionPathTree> predicate) {
      return firstNode.anyExecutionPathContains(predicate);
    }

    @Override
    public boolean allExecutionPathsContain(Predicate<ChainExecutionPathTree> predicate) {
      return firstNode.allExecutionPathsContain(predicate);
    }
  }

  private static class ChainNode implements ChainExecutionPathTree {

    private final ChainExecutionPathTree delegate;
    public Optional<ChainNode> next = empty();

    private ChainNode(ChainExecutionPathTree delegate) {
      this.delegate = delegate;
    }

    @Override
    public void accept(ChainExecutionPathTreeVisitor visitor) {
      this.delegate.accept(visitor);
      this.next.ifPresent(node -> node.accept(visitor));
    }

    @Override
    public ComponentAst getComponentAst() {
      return this.delegate.getComponentAst();
    }

    @Override
    public boolean anyExecutionPathContains(Predicate<ChainExecutionPathTree> predicate) {
      return this.delegate.anyExecutionPathContains(predicate)
          || this.next.map(node -> node.anyExecutionPathContains(predicate)).orElse(false);
    }

    @Override
    public boolean allExecutionPathsContain(Predicate<ChainExecutionPathTree> predicate) {
      return this.delegate.allExecutionPathsContain(predicate)
          || this.next.map(node -> node.allExecutionPathsContain(predicate)).orElse(false);
    }

  }

}
