/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.cfg.api;

import static java.util.Collections.emptyList;

import org.mule.api.annotation.Experimental;
import org.mule.runtime.ast.api.ComponentAst;

import java.util.List;
import java.util.function.Predicate;

/**
 * Control Flow Graph (Or execution path tree) that models the execution of a flow, Mule SDK Operation, entire application, or
 * subchain within a Mule Application. This tree can be traversed using the {@link ChainExecutionPathTreeVisitor}.
 * 
 * @since 1.1.
 */
@Experimental
public interface ChainExecutionPathTree {

  /**
   * Navigates this tree.
   * <p>
   * For each item of interest in this tree, a method from the provided {@link ChainExecutionPathTreeVisitor} will be called. The
   * order of the called methods is consistent and depends on the definition of the chain.
   * <p>
   * The provided {@code visitor} is expected to keep internal state for storing the results of the navigation. So, this method is
   * thread safe but the {@link ChainExecutionPathTreeVisitor} need not be (nor should be reused, even if used sequentially).
   *
   * @param visitor a visitor that will have its methods called as different items are found while navigating the tree.
   */
  void accept(ChainExecutionPathTreeVisitor visitor);

  /**
   * @return the {@link ComponentAst} that this subtree of the tree represents.
   */
  ComponentAst getComponentAst();

  /**
   * @param predicate
   * @return if any possible path of the execution has a {@link ComponentAst} that makes the {@param predicate} return
   *         {@code true}.
   */
  boolean anyExecutionPathContains(Predicate<ChainExecutionPathTree> predicate);

  /**
   * @param predicate
   * @return if all the possible paths of the execution have a {@link ComponentAst} that makes the {@param predicate} return
   *         {@code true}.
   */
  boolean allExecutionPathsContain(Predicate<ChainExecutionPathTree> predicate);

  /**
   * @param predicate
   * @return if none of the possible paths of the execution have a {@link ComponentAst} that makes the {@param predicate}
   *         {@code true}.
   */
  default boolean noneExecutionPathsContains(Predicate<ChainExecutionPathTree> predicate) {
    return !anyExecutionPathContains(predicate);
  }

  /**
   * @return a {@link List} with the children of this {@link ChainExecutionPathTree}.
   * @since 1.4
   */
  default List<ChainExecutionPathTree> children() {
    return emptyList();
  }

}
