/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.cfg.internal.node;

import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.cfg.api.ChainExecutionPathTree;
import org.mule.runtime.cfg.api.ChainExecutionPathTreeVisitor;

import java.util.function.Predicate;

public class ReferencedChainNode implements ChainExecutionPathTree {

  // ComponentAst of the element that references another tree.
  private final ComponentAst referencingAst;
  private final LazyValue<ChainExecutionPathTree> referenced;
  // to avoid infinite loops/recursions
  private boolean beingIterated;

  public ReferencedChainNode(ComponentAst componentAst, LazyValue<ChainExecutionPathTree> ref) {
    this.referencingAst = componentAst;
    this.referenced = ref;
  }

  @Override
  public boolean anyExecutionPathContains(Predicate<ChainExecutionPathTree> predicate) {
    if (beingIterated) {
      return false;
    }
    beingIterated = true;
    boolean result = predicate.test(this) || this.referenced.get().anyExecutionPathContains(predicate);
    beingIterated = false;
    return result;
  }

  @Override
  public boolean allExecutionPathsContain(Predicate<ChainExecutionPathTree> predicate) {
    if (beingIterated) {
      return false;
    }
    beingIterated = true;
    boolean result = predicate.test(this) || this.referenced.get().allExecutionPathsContain(predicate);
    beingIterated = false;
    return result;
  }

  @Override
  public void accept(ChainExecutionPathTreeVisitor visitor) {
    visitor.visitReferencedOperation(this);
    if (beingIterated) {
      visitor.finishReferencedOperation(this);
      return;
    }
    beingIterated = true;
    this.referenced.get().accept(visitor);
    beingIterated = false;
    visitor.finishReferencedOperation(this);
  }

  @Override
  public ComponentAst getComponentAst() {
    return referencingAst;
  }
}
