/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.cfg.internal.node;

import static java.util.Collections.singletonList;
import static java.util.Optional.empty;
import static org.mule.runtime.cfg.internal.node.NodeUtils.getErrorHandlers;
import static org.mule.runtime.cfg.internal.node.NodeUtils.isChainAlwaysExecuted;

import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.cfg.api.ChainExecutionPathTree;
import org.mule.runtime.cfg.api.ChainExecutionPathTreeVisitor;
import org.mule.runtime.cfg.internal.node.errorhandling.ErrorHandlingContext;

import java.util.List;
import java.util.Optional;
import java.util.function.Predicate;

public class ScopeExecutionPathNodeBuilder {

  private ComponentAst scopeAst;
  private ChainExecutionPathTree child;
  private Optional<ChainExecutionPathTree> errorHandlers = empty();

  public ScopeExecutionPathNodeBuilder(ComponentAst ast) {
    this.scopeAst = ast;
  }

  public ScopeExecutionPathNodeBuilder withErrorHandlerContext(ErrorHandlingContext context) {
    this.errorHandlers = getErrorHandlers(scopeAst, context);
    return this;
  }

  public ScopeExecutionPathNodeBuilder withChild(ChainExecutionPathTree child) {
    this.child = child;
    return this;
  }

  public ChainExecutionPathTree build() {
    return new ScopeNode(scopeAst, child, errorHandlers);
  }

  private static class ScopeNode extends AbstractWithErrorsNode {

    private final ComponentAst scopeAst;
    private final ChainExecutionPathTree child;

    private ScopeNode(ComponentAst componentAst, ChainExecutionPathTree child, Optional<ChainExecutionPathTree> errorHandlers) {
      super(errorHandlers);
      this.scopeAst = componentAst;
      this.child = child;
    }

    @Override
    protected void doAccept(ChainExecutionPathTreeVisitor visitor) {
      if (visitor.visitScope(this)) {
        child.accept(visitor);
      }
      visitor.scopeFinished(this);
    }

    @Override
    public ComponentAst getComponentAst() {
      return scopeAst;
    }

    @Override
    public boolean anyExecutionPathContains(Predicate<ChainExecutionPathTree> predicate) {
      return predicate.test(this) || this.child.anyExecutionPathContains(predicate);
    }

    @Override
    public boolean allExecutionPathsContain(Predicate<ChainExecutionPathTree> predicate) {
      return predicate.test(this)
          || (isChainAlwaysExecuted(this.child.getComponentAst()) && this.child.allExecutionPathsContain(predicate));
    }

    @Override
    public List<ChainExecutionPathTree> children() {
      return singletonList(this.child);
    }
  }
}
