/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.cfg.internal.node.errorhandling;

import static java.util.Collections.emptyList;
import static java.util.Collections.singleton;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;

import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.cfg.api.ChainExecutionPathTree;
import org.mule.runtime.cfg.api.ChainExecutionPathTreeVisitor;
import org.mule.runtime.api.message.error.matcher.ErrorTypeMatcher;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

public class ErrorHandlingExecutionPathNodeBuilder {

  private ComponentAst componentAst;
  private Optional<ChainExecutionPathTree> child = empty();
  private Optional<ErrorTypeMatcher> errorMatch = empty();

  public ErrorHandlingExecutionPathNodeBuilder(ComponentAst ast) {
    this.componentAst = ast;
  }

  public ErrorHandlingExecutionPathNodeBuilder setChild(ChainExecutionPathTree child) {
    this.child = of(child);
    return this;
  }

  public ErrorHandlingExecutionPathNodeBuilder setErrorMatcher(ErrorTypeMatcher errorExpression) {
    this.errorMatch = of(errorExpression);
    return this;
  }

  public ErrorHandlerNode build() {
    return new ErrorHandlingImplNode(componentAst, child, errorMatch);
  }

  private static class ErrorHandlingImplNode implements ErrorHandlerNode {

    private final ComponentAst componentAst;
    private final Optional<ChainExecutionPathTree> child;
    private final Optional<ErrorTypeMatcher> errorMatch;
    private List<Pair<ErrorHandlerNode, Set<ErrorType>>> outers = new ArrayList<>();

    private ErrorHandlingImplNode(ComponentAst componentAst, Optional<ChainExecutionPathTree> child,
                                  Optional<ErrorTypeMatcher> errorMatch) {
      this.componentAst = componentAst;
      this.child = child;
      this.errorMatch = errorMatch;
    }

    @Override
    public boolean anyExecutionPathContains(Predicate<ChainExecutionPathTree> predicate) {
      return predicate.test(this) || this.child.map(c -> c.anyExecutionPathContains(predicate)).orElse(false);
    }

    @Override
    public boolean allExecutionPathsContain(Predicate<ChainExecutionPathTree> predicate) {
      return predicate.test(this) || this.child.map(c -> c.allExecutionPathsContain(predicate)).orElse(true);
    }

    @Override
    public boolean appliesTo(ErrorType errorType) {
      return this.errorMatch.map(em -> em.match(errorType)).orElse(true);
    }

    @Override
    public void setOuters(List<ErrorHandlerNode> outer) {
      this.outers = outer.stream()
          .map(o -> new Pair<>(o, (Set<ErrorType>) new HashSet<ErrorType>()))
          .collect(toList());
    }

    @Override
    public void accept(ChainExecutionPathTreeVisitor visitor, ErrorType incoming) {
      visitor.errorHandlerCouldBeInvoked(this, incoming);
      outers.stream()
          .filter(outer -> outer.getFirst().appliesTo(incoming))
          .findFirst()
          .ifPresent(outer -> outer.getSecond().add(incoming));
    }

    @Override
    public void accept(ChainExecutionPathTreeVisitor visitor) {
      if (!visitor.errorHandlerStarted(this)) {
        return;
      }
      child.ifPresent(c -> c.accept(visitor));
      visitor.errorHandlerFinished(this);
      outers.forEach(outer -> outer.getSecond()
          .forEach(errorType -> visitor.errorHandlerCouldBeInvoked(outer.getFirst(), errorType)));
      visitor.errorHandlerPropagationComplete(this);
    }

    @Override
    public ComponentAst getComponentAst() {
      return this.componentAst;
    }

    @Override
    public List<ChainExecutionPathTree> children() {
      return this.child.map(Collections::singletonList).orElse(emptyList());
    }
  }

}
