/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.cfg.utils;

import static org.mule.runtime.api.component.ComponentIdentifier.buildFromStringRepresentation;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.ERROR_HANDLER;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.FLOW;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.ON_ERROR;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.OPERATION;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.ROUTE;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.ROUTER;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.SCOPE;
import static org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED;
import static org.mule.runtime.api.meta.model.nested.ChainExecutionOccurrence.ONCE;
import static org.mule.runtime.api.meta.model.nested.ChainExecutionOccurrence.ONCE_OR_NONE;
import static org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.DEFAULT_GROUP_NAME;
import static org.mule.runtime.ast.api.ComponentMetadataAst.EMPTY_METADATA;
import static org.mule.runtime.ast.api.error.ErrorTypeBuilder.builder;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.Collections.singletonList;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.metadata.api.ClassTypeLoader;
import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.TypedComponentIdentifier;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.api.meta.MuleVersion;
import org.mule.runtime.api.meta.model.ComponentModelVisitor;
import org.mule.runtime.api.meta.model.ComponentVisibility;
import org.mule.runtime.api.meta.model.ModelProperty;
import org.mule.runtime.api.meta.model.deprecated.DeprecationModel;
import org.mule.runtime.api.meta.model.display.DisplayModel;
import org.mule.runtime.api.meta.model.error.ErrorModel;
import org.mule.runtime.api.meta.model.error.ImmutableErrorModel;
import org.mule.runtime.api.meta.model.nested.ChainExecutionOccurrence;
import org.mule.runtime.api.meta.model.nested.NestableElementModel;
import org.mule.runtime.api.meta.model.nested.NestableElementModelVisitor;
import org.mule.runtime.api.meta.model.nested.NestedChainModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.meta.model.stereotype.StereotypeModel;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentGenerationInformation;
import org.mule.runtime.ast.api.builder.ComponentAstBuilder;
import org.mule.runtime.ast.api.error.ErrorTypeBuilder;
import org.mule.runtime.ast.internal.DefaultComponentParameterAst;
import org.mule.runtime.ast.internal.builder.DefaultComponentAstBuilder;
import org.mule.runtime.ast.internal.builder.PropertiesResolver;
import org.mule.runtime.ast.internal.model.DefaultExtensionModelHelper;
import org.mule.runtime.cfg.api.ChainExecutionPathTree;
import org.mule.runtime.cfg.api.ChainExecutionPathTreeFactory;
import org.mule.runtime.cfg.internal.queries.ErrorHandlingQueriesTestCase;
import org.mule.runtime.extension.api.declaration.type.ExtensionsTypeLoaderFactory;
import org.mule.sdk.api.stereotype.MuleStereotypes;

import org.mockito.stubbing.Answer;

import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

public final class CfgTestUtils {

  public static final ComponentIdentifier FLOW_COMP = buildFromStringRepresentation("flow");
  public static final ComponentIdentifier SET_PAYLODAD = buildFromStringRepresentation("set-payload");
  public static final ComponentIdentifier SET_VARIABLE = buildFromStringRepresentation("set-variable");
  public static final ComponentIdentifier LOGGER = buildFromStringRepresentation("logger");
  public static final ComponentIdentifier FOREACH = buildFromStringRepresentation("foreach");
  public static final ComponentIdentifier NOOP = buildFromStringRepresentation("noop");
  public static final ComponentIdentifier NOOP2 = buildFromStringRepresentation("noop2");
  public static final ComponentIdentifier TRYSCOPE = buildFromStringRepresentation("try");
  public static final ComponentIdentifier SCATTER_GATHER = buildFromStringRepresentation("scatter-gather");
  public static final ComponentIdentifier CHOICE = buildFromStringRepresentation("choice");
  public static final ComponentIdentifier PARALLEL_FOREACH = buildFromStringRepresentation("parallel-foreach");
  public static final ComponentIdentifier DB_SELECT = ComponentIdentifier.builder().namespace("db").name("select").build();
  public static final ComponentIdentifier JMS_PUBLISH = ComponentIdentifier.builder().namespace("jms").name("publish").build();
  public static final ComponentIdentifier ROUTE_ELEMENT = buildFromStringRepresentation("route");
  public static final ComponentIdentifier ERROR_HANDLER_COMPONENT = buildFromStringRepresentation("error-handler");
  public static final ComponentIdentifier ERROR_CONTINUE = buildFromStringRepresentation("on-error-continue");
  public static final ComponentIdentifier ERROR_PROPAGATE = buildFromStringRepresentation("on-error-propagate");

  private CfgTestUtils() {

  }

  /**
   * Structure of this Ast: <code>
   * - Flow
   *    - Logger
   *    - Foreach
   *        - Logger
   *        - Scatter-Gather
   *            - Route
   *                - Set-payload
   *            - Route
   *                - db:select
   *        - Logger
   *    - Choice
   *        - Route
   *            - Set-payload
   *        - Route
   *            - Set-Variable
   *            - Logger
   *     - Try scope
   *         - noop
   * </code>
   */
  public static ChainExecutionPathTree testTree() {
    final PropertiesResolver propertiesResolver = new PropertiesResolver();
    final DefaultComponentAstBuilder compBuilder =
        new DefaultComponentAstBuilder(propertiesResolver, getExtensionModelHelper(), emptyList(), 0);
    compBuilder.withIdentifier(FLOW_COMP).withRawParameter("name", "test")
        .withMetadata(EMPTY_METADATA);
    compBuilder.addChildComponent().withIdentifier(LOGGER).withMetadata(EMPTY_METADATA);
    final DefaultComponentAstBuilder foreach = (DefaultComponentAstBuilder) compBuilder.addChildComponent()
        .withIdentifier(FOREACH).withMetadata(EMPTY_METADATA).withParameterizedModel(new MockChainModel(ONCE_OR_NONE));
    foreach.withComponentType(SCOPE);
    foreach.addChildComponent().withIdentifier(LOGGER).withMetadata(EMPTY_METADATA);
    final ComponentAstBuilder scatter = foreach.addChildComponent()
        .withIdentifier(SCATTER_GATHER).withMetadata(EMPTY_METADATA);
    scatter.addChildComponent().withIdentifier(ROUTE_ELEMENT).withMetadata(EMPTY_METADATA)
        .addChildComponent().withIdentifier(SET_PAYLODAD).withMetadata(EMPTY_METADATA);
    scatter.addChildComponent().withIdentifier(ROUTE_ELEMENT).withMetadata(EMPTY_METADATA)
        .addChildComponent().withIdentifier(DB_SELECT).withMetadata(EMPTY_METADATA);
    foreach.addChildComponent().withIdentifier(LOGGER).withMetadata(EMPTY_METADATA);
    final ComponentAstBuilder choice =
        compBuilder.addChildComponent().withIdentifier(CHOICE).withMetadata(EMPTY_METADATA);
    choice.addChildComponent().withIdentifier(ROUTE_ELEMENT).withMetadata(EMPTY_METADATA)
        .addChildComponent().withIdentifier(SET_PAYLODAD).withMetadata(EMPTY_METADATA);
    final ComponentAstBuilder otherwiseRoute =
        choice.addChildComponent().withIdentifier(ROUTE_ELEMENT).withMetadata(EMPTY_METADATA);
    otherwiseRoute.addChildComponent().withIdentifier(SET_VARIABLE).withMetadata(EMPTY_METADATA);
    otherwiseRoute.addChildComponent().withIdentifier(LOGGER).withMetadata(EMPTY_METADATA);

    final DefaultComponentAstBuilder tryscope =
        (DefaultComponentAstBuilder) compBuilder.addChildComponent().withIdentifier(TRYSCOPE).withMetadata(EMPTY_METADATA);
    tryscope.withComponentType(SCOPE).withParameterizedModel(new MockChainModel(ONCE)).withMetadata(EMPTY_METADATA);
    tryscope.addChildComponent().withIdentifier(NOOP).withMetadata(EMPTY_METADATA);

    return new ChainExecutionPathTreeFactory(mock(ArtifactAst.class)).generateFor(compBuilder.build());
  }

  private static DefaultExtensionModelHelper getExtensionModelHelper() {
    DefaultExtensionModelHelper helper = mock(DefaultExtensionModelHelper.class);
    when(helper.findComponentType(any(ComponentIdentifier.class)))
        .thenAnswer((Answer<TypedComponentIdentifier.ComponentType>) invocationOnMock -> {
          ComponentIdentifier identifier = invocationOnMock.getArgument(0);
          if (identifier.equals(FLOW_COMP)) {
            return FLOW;
          } else if (identifier.equals(FOREACH) || identifier.equals(TRYSCOPE) || identifier.equals(PARALLEL_FOREACH)) {
            return SCOPE;
          } else if (identifier.equals(CHOICE) || identifier.equals(SCATTER_GATHER)) {
            return ROUTER;
          } else if (identifier.equals(ROUTE_ELEMENT)) {
            return ROUTE;
          } else if (identifier.equals(ERROR_HANDLER_COMPONENT)) {
            return ERROR_HANDLER;
          } else if (identifier.equals(ERROR_CONTINUE) || identifier.equals(ERROR_PROPAGATE)) {
            return ON_ERROR;
          } else {
            return OPERATION;
          }
        });
    return helper;
  }

  /**
   * Structure of this Ast: <code>
   * - Flow
   *    - Logger
   *    - Try
   *        - Logger
   *        - Choice
   *            - Route
   *                - Set-Payload
   *            - Route
   *                - db:select (errors: SOME and OTHER)
   *        - Logger
   *        - Error Handler
   *            - On error propagate (type TEST:SOME)
   *                - Set-variable
   *                - JMS Publish
   *                - Noop (no-operation)
   *            - On error continue  WITH or WITHOUT TEST:OTHER depending on the parameter
   *                - Set-variable
   *    - Error Handler
   *        - On error continue (type MULE:NOTHING)
   *           - Noop2 (another no-operation)
   *        - On error continue (no type)
   *            - set-variable
   * </code>
   */
  public static ChainExecutionPathTree testTreeWithErrorHandling(boolean continueWithAllErrors) {
    final PropertiesResolver propertiesResolver = new PropertiesResolver();
    final DefaultComponentAstBuilder compBuilder =
        new DefaultComponentAstBuilder(propertiesResolver, getExtensionModelHelper(), emptyList(), 0);
    compBuilder.withIdentifier(FLOW_COMP).withRawParameter("name", "test")
        .withMetadata(EMPTY_METADATA);
    addFlowNameParameter(compBuilder);
    compBuilder.addChildComponent().withIdentifier(LOGGER).withMetadata(EMPTY_METADATA);
    final DefaultComponentAstBuilder tryScope = (DefaultComponentAstBuilder) compBuilder.addChildComponent()
        .withIdentifier(TRYSCOPE).withMetadata(EMPTY_METADATA);
    tryScope.addChildComponent().withIdentifier(LOGGER)
        .withMetadata(EMPTY_METADATA);
    final ComponentAstBuilder choice = tryScope.addChildComponent()
        .withIdentifier(CHOICE).withMetadata(EMPTY_METADATA);
    choice.addChildComponent().withIdentifier(ROUTE_ELEMENT).withMetadata(EMPTY_METADATA)
        .addChildComponent().withIdentifier(SET_PAYLODAD).withMetadata(EMPTY_METADATA);
    final DefaultComponentAstBuilder select =
        (DefaultComponentAstBuilder) choice.addChildComponent().withIdentifier(ROUTE_ELEMENT).withMetadata(EMPTY_METADATA)
            .addChildComponent().withIdentifier(DB_SELECT).withMetadata(EMPTY_METADATA);
    addErrors(select, "SOME", "OTHER");
    select.withComponentType(OPERATION);
    tryScope.addChildComponent().withIdentifier(LOGGER).withMetadata(EMPTY_METADATA);
    DefaultComponentAstBuilder errorHandler = (DefaultComponentAstBuilder) tryScope.addChildComponent()
        .withIdentifier(ERROR_HANDLER_COMPONENT)
        .withMetadata(EMPTY_METADATA);
    addErrorHandlerParameterizedModel(errorHandler);
    DefaultComponentAstBuilder propagateEH =
        (DefaultComponentAstBuilder) errorHandler.addChildComponent().withIdentifier(ERROR_PROPAGATE)
            .withMetadata(EMPTY_METADATA);
    addErrorHandlingType(propagateEH, "TEST:SOME");
    propagateEH.addChildComponent().withIdentifier(SET_VARIABLE).withMetadata(EMPTY_METADATA);
    propagateEH.addChildComponent().withIdentifier(JMS_PUBLISH).withMetadata(EMPTY_METADATA);
    propagateEH.addChildComponent().withIdentifier(NOOP).withMetadata(EMPTY_METADATA);
    propagateEH.withComponentType(ON_ERROR);
    DefaultComponentAstBuilder continueEH =
        (DefaultComponentAstBuilder) errorHandler.addChildComponent().withIdentifier(ERROR_CONTINUE).withMetadata(EMPTY_METADATA);
    continueEH.addChildComponent().withIdentifier(SET_VARIABLE).withMetadata(EMPTY_METADATA);
    continueEH.withComponentType(ON_ERROR);
    addErrorHandlingType(continueEH, continueWithAllErrors ? null : "TEST:OTHER");
    errorHandler = (DefaultComponentAstBuilder) compBuilder.addChildComponent()
        .withIdentifier(ERROR_HANDLER_COMPONENT)
        .withMetadata(EMPTY_METADATA);
    addErrorHandlerParameterizedModel(errorHandler);
    continueEH =
        (DefaultComponentAstBuilder) errorHandler.addChildComponent().withIdentifier(ERROR_CONTINUE).withMetadata(EMPTY_METADATA);
    continueEH.addChildComponent().withIdentifier(NOOP2).withMetadata(EMPTY_METADATA);
    continueEH.withComponentType(ON_ERROR);
    addErrorHandlingType(continueEH, "MULE:NOTHING");
    continueEH =
        (DefaultComponentAstBuilder) errorHandler.addChildComponent().withIdentifier(ERROR_CONTINUE).withMetadata(EMPTY_METADATA);
    continueEH.addChildComponent().withIdentifier(SET_VARIABLE).withMetadata(EMPTY_METADATA);
    continueEH.withComponentType(ON_ERROR);
    addErrorHandlingType(continueEH, null);
    return new ChainExecutionPathTreeFactory(getApplication()).generateFor(compBuilder.build());
  }

  private static ParameterGroupModel createMockParameterGroup() {
    ParameterGroupModel groupModel = mock(ParameterGroupModel.class);
    when(groupModel.getName()).thenReturn(DEFAULT_GROUP_NAME);
    when(groupModel.getParameterModels()).thenReturn(emptyList());
    return groupModel;
  }

  private static void addErrorHandlerParameterizedModel(DefaultComponentAstBuilder builder) {
    ParameterizedModel model = mock(ParameterizedModel.class);
    builder.withParameterizedModel(model);
    final ClassTypeLoader typeLoader = ExtensionsTypeLoaderFactory.getDefault()
        .createTypeLoader(CfgTestUtils.class.getClassLoader());
    final MetadataType stringType = typeLoader.load(String.class);
    ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getName()).thenReturn("ref");
    when(parameterModel.getType()).thenReturn(stringType);
    when(parameterModel.isComponentId()).thenReturn(true);
    when(parameterModel.getExpressionSupport()).thenReturn(NOT_SUPPORTED);
    when(parameterModel.getAllowedStereotypes()).thenReturn(singletonList(MuleStereotypes.ERROR_HANDLER));

    ParameterGroupModel defaultGroup = createMockParameterGroup();
    when(defaultGroup.getParameterModels()).thenReturn(singletonList(parameterModel));
    when(defaultGroup.getParameter("ref")).thenReturn(of(parameterModel));

    when(model.getAllParameterModels()).thenReturn(singletonList(parameterModel));
    when(model.getParameterGroupModels()).thenReturn(singletonList(defaultGroup));

    builder.withComponentType(ERROR_HANDLER);
  }

  private static void addErrorHandlingType(DefaultComponentAstBuilder errorhandler, String type) {
    ClassTypeLoader typeLoader =
        ExtensionsTypeLoaderFactory.getDefault().createTypeLoader(ErrorHandlingQueriesTestCase.class.getClassLoader());
    MetadataType stringType = typeLoader.load(String.class);
    ParameterModel typeModel = mock(ParameterModel.class);
    when(typeModel.getName()).thenReturn("type");
    when(typeModel.getType()).thenReturn(stringType);
    ParameterGroupModel defaultGroup = createMockParameterGroup();
    errorhandler.withParameter(typeModel, defaultGroup,
                               new DefaultComponentParameterAst(type, typeModel, defaultGroup,
                                                                mock(ComponentGenerationInformation.class),
                                                                new PropertiesResolver()),
                               empty());

    ParameterizedModel parameterized = mock(ParameterizedModel.class);
    when(parameterized.getAllParameterModels()).thenReturn(singletonList(typeModel));
    when(parameterized.getParameterGroupModels()).thenReturn(singletonList(defaultGroup));
    errorhandler.withParameterizedModel(parameterized);
  }

  private static void addErrors(DefaultComponentAstBuilder builder, String... errors) {
    OperationModel model = mock(OperationModel.class);
    Set<ErrorModel> errorsModels = new HashSet<>();
    ErrorModel universalParent = new ImmutableErrorModel("ANY", "MULE", true, null);
    for (String err : errors) {
      errorsModels.add(new ImmutableErrorModel(err, "TEST", true, universalParent));
    }
    when(model.getErrorModels()).thenReturn(errorsModels);
    builder.withComponentModel(model);
  }

  private static ArtifactAst getApplication() {
    ArtifactAst application = mock(ArtifactAst.class);
    ErrorTypeRepository repository = mock(ErrorTypeRepository.class);

    ErrorType parent = builder().identifier("ANY").namespace("MULE").build();

    when(repository.lookupErrorType(any())).thenAnswer((Answer<Optional<ErrorType>>) invocationOnMock -> {
      ComponentIdentifier identifier = invocationOnMock.getArgument(0);
      ErrorTypeBuilder builder = ErrorTypeBuilder.builder().identifier(identifier.getName()).namespace(identifier.getNamespace());
      if (!identifier.getName().equals("ANY") || !identifier.getNamespace().equals("MULE")) {
        builder.parentErrorType(parent);
      }
      return of(builder.build());
    });
    when(application.getErrorTypeRepository()).thenReturn(repository);
    return application;
  }

  private static void addFlowNameParameter(DefaultComponentAstBuilder builder) {
    ParameterizedModel model = mock(ParameterizedModel.class);
    builder.withParameterizedModel(model);
    final ClassTypeLoader typeLoader = ExtensionsTypeLoaderFactory.getDefault()
        .createTypeLoader(CfgTestUtils.class.getClassLoader());
    final MetadataType stringType = typeLoader.load(String.class);
    ParameterModel flatParameterModel = mock(ParameterModel.class);
    when(flatParameterModel.getName()).thenReturn("flowName");
    when(flatParameterModel.getType()).thenReturn(stringType);
    when(flatParameterModel.isComponentId()).thenReturn(true);
    when(model.getAllParameterModels()).thenReturn(singletonList(flatParameterModel));

    ParameterGroupModel defaultGroup = mock(ParameterGroupModel.class);
    when(defaultGroup.isShowInDsl()).thenReturn(false);
    when(defaultGroup.getName()).thenReturn("Default");
    when(defaultGroup.getParameterModels()).thenReturn(singletonList(flatParameterModel));
    when(defaultGroup.getParameter("flowName")).thenReturn(of(flatParameterModel));

    builder.withParameter(flatParameterModel, defaultGroup,
                          new DefaultComponentParameterAst("name", flatParameterModel, defaultGroup,
                                                           mock(ComponentGenerationInformation.class), new PropertiesResolver()),
                          empty());
    builder.withComponentType(FLOW);
  }

  public static boolean hasIdentifier(ChainExecutionPathTree tree, ComponentIdentifier identifier) {
    return tree.getComponentAst().getIdentifier().equals(identifier);
  }

  private static class MockChainModel implements NestedChainModel {

    private final ChainExecutionOccurrence occurrence;

    public MockChainModel(ChainExecutionOccurrence occurrence) {
      this.occurrence = occurrence;
    }


    @Override
    public boolean isRequired() {
      return false;
    }

    @Override
    public Set<StereotypeModel> getAllowedStereotypes() {
      return null;
    }

    @Override
    public int getMinOccurs() {
      return 1;
    }

    @Override
    public Optional<Integer> getMaxOccurs() {
      return Optional.of(1);
    }

    @Override
    public void accept(NestableElementModelVisitor visitor) {

    }

    @Override
    public void accept(ComponentModelVisitor visitor) {

    }

    @Override
    public ComponentVisibility getVisibility() {
      return null;
    }

    @Override
    public List<? extends NestableElementModel> getNestedComponents() {
      return emptyList();
    }

    @Override
    public <T extends ModelProperty> Optional<T> getModelProperty(Class<T> propertyType) {
      return empty();
    }

    @Override
    public Set<ModelProperty> getModelProperties() {
      return emptySet();
    }

    @Override
    public Optional<DeprecationModel> getDeprecationModel() {
      return empty();
    }

    @Override
    public Optional<DisplayModel> getDisplayModel() {
      return empty();
    }

    @Override
    public Set<ErrorModel> getErrorModels() {
      return emptySet();
    }

    @Override
    public ChainExecutionOccurrence getChainExecutionOccurrence() {
      return occurrence;
    }

    @Override
    public List<ParameterGroupModel> getParameterGroupModels() {
      return emptyList();
    }

    @Override
    public String getDescription() {
      return null;
    }

    @Override
    public String getName() {
      return null;
    }

    @Override
    public StereotypeModel getStereotype() {
      return null;
    }

    @Override
    public Optional<MuleVersion> getMinMuleVersion() {
      return empty();
    }
  }
}
