/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.cfg.visitors;

import org.mule.runtime.api.component.TypedComponentIdentifier;
import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.cfg.api.ChainExecutionPathTree;
import org.mule.runtime.cfg.api.ChainExecutionPathTreeVisitor;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Deque;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * Retrieves all the possible execution paths while visiting the {@link org.mule.runtime.cfg.api.ChainExecutionPathTree}.
 */
public class AllExecutionPathsVisitor implements ChainExecutionPathTreeVisitor {

  private final Deque<List<List<ComponentAst>>> executions = new ArrayDeque<>();

  public AllExecutionPathsVisitor() {
    addEmptyExecution();
  }

  private void add(ComponentAst ast) {
    for (List<ComponentAst> possibleExecutions : executions.peek()) {
      if (shouldMerge(possibleExecutions, ast)) {
        possibleExecutions.add(ast);
      }
    }
  }

  public List<List<ComponentAst>> getAllPossibleExecutions() {
    if (executions.size() != 1) {
      throw new IllegalStateException("Executions should have finished with just one flattened list with all the possibilities");
    }
    return executions.peek();
  }

  @Override
  public void visitSource(ChainExecutionPathTree source) {
    add(source.getComponentAst());
  }

  @Override
  public void visitSimpleOperation(ChainExecutionPathTree operation) {
    add(operation.getComponentAst());
  }

  @Override
  public void visitReferencedOperation(ChainExecutionPathTree operation) {

  }

  @Override
  public void finishReferencedOperation(ChainExecutionPathTree operation) {

  }

  @Override
  public boolean visitScope(ChainExecutionPathTree scope) {
    add(scope.getComponentAst());
    addEmptyExecution();
    return true;
  }

  @Override
  public boolean visitRouter(ChainExecutionPathTree router) {
    add(router.getComponentAst());
    return true;
  }

  @Override
  public boolean innerRouteStarted(ChainExecutionPathTree innerChain) {
    addEmptyExecution();
    return true;
  }

  @Override
  public void innerRouteFinished(ChainExecutionPathTree innerChain) {

  }

  @Override
  public void scopeFinished(ChainExecutionPathTree scope) {
    finishedContainer(1);
  }

  @Override
  public void routerFinished(ChainExecutionPathTree router) {
    finishedContainer(router.getComponentAst().directChildren().size());
  }

  @Override
  public void errorHandlerCouldBeInvoked(ChainExecutionPathTree errorHandler, ErrorType errorType) {

  }

  private boolean shouldMerge(List<ComponentAst> previous, ComponentAst nextFirst) {
    if (previous.isEmpty()) {
      return true;
    }
    ComponentAst previousLast = previous.get(previous.size() - 1);
    if (previousLast.getComponentType() != TypedComponentIdentifier.ComponentType.ON_ERROR) {
      return true;
    }
    if (!previousLast.getLocation().getRootContainerName().equals(nextFirst.getLocation().getRootContainerName())) {
      return true;
    }
    return !nextFirst.getLocation().getLocation().contains(previousLast.getLocation().getLocation().split("errorHandler")[0]);
  }

  private boolean shouldMerge(List<ComponentAst> previous, List<ComponentAst> next) {
    if (previous.isEmpty() || next.isEmpty()) {
      return true;
    }
    return shouldMerge(previous, next.get(0));
  }

  private void finishedContainer(int branches) {
    List<List<List<ComponentAst>>> branchedExecutions = new ArrayList<>();
    for (int i = 0; i < branches; i++) {
      branchedExecutions.add(executions.pop());
    }
    Set<Integer> addedIndexesAlone = new HashSet<>();

    List<List<ComponentAst>> mergedPeek = new ArrayList<>();
    List<List<ComponentAst>> possibleExecutions = executions.pop();
    for (int i = 0; i < possibleExecutions.size(); i++) {
      for (List<List<ComponentAst>> sub : branchedExecutions) {
        for (List<ComponentAst> newPossible : sub) {
          List<ComponentAst> newRoute = new ArrayList<>();
          newRoute.addAll(possibleExecutions.get(i));
          if (shouldMerge(possibleExecutions.get(i), newPossible)) {
            newRoute.addAll(newPossible);
            mergedPeek.add(newRoute);
          } else {
            if (!addedIndexesAlone.contains(i)) {
              addedIndexesAlone.add(i);
              mergedPeek.add(newRoute);
            }
          }
        }
      }
    }
    executions.push(mergedPeek);
  }

  void addEmptyExecution() {
    List<List<ComponentAst>> init = new ArrayList<>();
    init.add(new ArrayList<>());
    executions.push(init);
  }


  @Override
  public boolean errorHandlerStarted(ChainExecutionPathTree handler) {
    return false;
  }

  @Override
  public void errorHandlerPropagationComplete(ChainExecutionPathTree handler) {}

  @Override
  public void errorHandlerFinished(ChainExecutionPathTree handler) {}

  @Override
  public void chainFinishedSuccess() {

  }

  @Override
  public void chainFinishedSuccessWithResponse(ChainExecutionPathTree source) {

  }

  @Override
  public void chainFinishedError() {

  }

  @Override
  public void chainFinishedErrorWithResponse(ChainExecutionPathTree source) {

  }
}
