/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.cfg.visitors;

import static org.hamcrest.CoreMatchers.hasItems;
import static org.mule.runtime.cfg.AllureCfgConstants.CFG_FEATURE;
import static org.mule.runtime.cfg.AllureCfgConstants.Cfg.VISITORS;
import static org.mule.runtime.cfg.utils.CfgTestUtils.testTree;
import static org.mule.runtime.cfg.utils.CfgTestUtils.testTreeWithErrorHandling;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.Ignore;
import org.junit.Test;
import org.mule.runtime.cfg.api.ChainExecutionPathTree;

@Feature(CFG_FEATURE)
@Story(VISITORS)
public class SimpleVisitingTestCase {

  @Test
  public void countOnSimpleApplication() {
    ChainExecutionPathTree tree = testTree();
    CountingVisitor visitor = new CountingVisitor();
    tree.accept(visitor);
    assertThat(visitor.sourceCount, is(0));
    assertThat(visitor.operationCount, is(9));
    assertThat(visitor.scopeCount, is(2));
    assertThat(visitor.routerCount, is(2));
    assertThat(visitor.errorHandlerCount, is(0));
  }

  @Test
  @Ignore("W-12450002")
  public void countOnApplicationWithErrorHandling() {
    ChainExecutionPathTree tree = testTreeWithErrorHandling(true);
    CountingVisitor visitor = new CountingVisitor();
    tree.accept(visitor);
    assertThat(visitor.sourceCount, is(0));
    assertThat(visitor.operationCount, is(11));
    assertThat(visitor.scopeCount, is(1));
    assertThat(visitor.routerCount, is(1));
    assertThat(visitor.errorHandlerCount, is(4));
    assertThat(visitor.potentialErrorInvocations.keySet(), hasItems("name/1/3/0", "name/1/3/1", "name/2/1"));
    assertThat(visitor.potentialErrorInvocations.get("name/1/3/0"), is(1));
    assertThat(visitor.potentialErrorInvocations.get("name/1/3/1"), is(1));
    assertThat(visitor.potentialErrorInvocations.get("name/2/1"), is(1));
  }

  @Test
  public void countOnApplicationWithErrorHandlingWithNoMatchingTypes() {
    ChainExecutionPathTree tree = testTreeWithErrorHandling(false);
    CountingVisitor visitor = new CountingVisitor();
    tree.accept(visitor);
    assertThat(visitor.sourceCount, is(0));
    assertThat(visitor.operationCount, is(11));
    assertThat(visitor.scopeCount, is(1));
    assertThat(visitor.routerCount, is(1));
    assertThat(visitor.errorHandlerCount, is(4));
    assertThat(visitor.potentialErrorInvocations.keySet(), hasItems("name/1/3/0", "name/1/3/1", "name/2/1"));
    assertThat(visitor.potentialErrorInvocations.get("name/1/3/0"), is(1));
    assertThat(visitor.potentialErrorInvocations.get("name/1/3/1"), is(1));
    assertThat(visitor.potentialErrorInvocations.get("name/2/1"), is(1));
  }

}
