/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.cfg.internal.node;

import static java.util.stream.Collectors.toList;

import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.ROUTE;
import static org.mule.runtime.cfg.internal.node.NodeUtils.getErrorHandlers;
import static org.mule.runtime.cfg.internal.node.NodeUtils.isChainAlwaysExecuted;

import static java.util.Collections.unmodifiableList;

import static java.util.Optional.empty;

import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.api.meta.model.nested.NestedRouteModel;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.cfg.api.ChainExecutionPathTree;
import org.mule.runtime.cfg.api.ChainExecutionPathTreeVisitor;
import org.mule.runtime.cfg.internal.node.errorhandling.ErrorHandlingContext;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.function.Predicate;
import java.util.stream.Stream;

public class RouterExecutionPathNodeBuilder {

  private final ComponentAst routerAst;
  private final List<ChainExecutionPathTree> children = new ArrayList<>();
  private Optional<ChainExecutionPathTree> errorHandlers = empty();

  public RouterExecutionPathNodeBuilder(ComponentAst ast) {
    this.routerAst = ast;
  }

  public RouterExecutionPathNodeBuilder withRoute(ChainExecutionPathTree child) {
    children.add(child);
    return this;
  }

  public RouterExecutionPathNodeBuilder withErrorHandlerContext(ErrorHandlingContext context) {
    this.errorHandlers = getErrorHandlers(routerAst, context);
    return this;
  }

  public ChainExecutionPathTree build() {
    return new RouterNode(routerAst, children, errorHandlers);
  }

  private static class RouterNode extends AbstractWithErrorsNode {

    private final ComponentAst routerAst;
    private final List<ChainExecutionPathTree> children;
    private final LazyValue<Optional<ErrorType>> alwaysInterruptsWithError;

    private RouterNode(ComponentAst componentAst, List<ChainExecutionPathTree> children,
                       Optional<ChainExecutionPathTree> errorHandlers) {
      super(errorHandlers);
      this.routerAst = componentAst;
      this.children = children.stream()
          .flatMap(child -> {
            if (ROUTE.equals(child.getComponentAst().getComponentType())
                && !child.getComponentAst().getModel(NestedRouteModel.class).isPresent()) {
              // list of routers
              return child.children().stream()
                  .filter(c -> c.getComponentAst().getModel(NestedRouteModel.class).isPresent());
            } else {
              return Stream.of(child);
            }
          })
          .collect(toList());
      this.alwaysInterruptsWithError = new LazyValue<>(this::computeAlwaysInterruptsWithError);
    }

    @Override
    protected void doAccept(ChainExecutionPathTreeVisitor visitor) {
      if (!visitor.visitRouter(this)) {
        visitor.routerFinished(this);
        return;
      }
      children.forEach(child -> {
        visitor.innerRouteStarted(child);
        child.accept(visitor);
        visitor.innerRouteFinished(child);
      });
      visitor.routerFinished(this);
    }

    @Override
    public ComponentAst getComponentAst() {
      return routerAst;
    }

    @Override
    public boolean anyExecutionPathContains(Predicate<ChainExecutionPathTree> predicate) {
      if (predicate.test(this)) {
        return true;
      }
      for (ChainExecutionPathTree child : children) {
        if (child.anyExecutionPathContains(predicate)) {
          return true;
        }
      }
      return false;
    }

    @Override
    public boolean allExecutionPathsContain(Predicate<ChainExecutionPathTree> predicate) {
      if (predicate.test(this)) {
        return true;
      }
      for (ChainExecutionPathTree child : children) {
        // If the route might not be executed, then we can't count it for "all possible executions"
        if (isChainAlwaysExecuted(child.getComponentAst())) {
          if (child.allExecutionPathsContain(predicate)) {
            return true;
          }
        }
      }
      return false;
    }

    @Override
    public List<ChainExecutionPathTree> children() {
      return unmodifiableList(this.children);
    }

    @Override
    public Optional<ErrorType> alwaysInterruptsWithError() {
      return alwaysInterruptsWithError.get();
    }

    private Optional<ErrorType> computeAlwaysInterruptsWithError() {
      for (ChainExecutionPathTree child : children) {
        if (isChainAlwaysExecuted(child.getComponentAst())) {
          Optional<ErrorType> chainError = NodeUtils.alwaysInterruptsWithError(child);
          if (chainError.isPresent()) {
            return chainError;
          }
        }
      }
      return empty();
    }
  }
}
