/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.cfg.internal.node;

import static org.mule.runtime.api.component.ComponentIdentifier.buildFromStringRepresentation;
import static org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.DEFAULT_GROUP_NAME;

import static java.lang.String.format;
import static java.util.Collections.singletonList;
import static java.util.Optional.empty;
import static java.util.Optional.of;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.cfg.api.ChainExecutionPathTree;
import org.mule.runtime.cfg.api.ChainExecutionPathTreeVisitor;
import org.mule.runtime.cfg.internal.node.errorhandling.ErrorHandlingContext;

import java.util.Optional;
import java.util.function.Predicate;

public class SimpleOperationNode extends AbstractWithErrorsNode {

  private static final ComponentIdentifier RAISE_ERROR_IDENTIFIER = ComponentIdentifier.builder()
      .namespace("mule")
      .name("raise-error").build();

  private final ComponentAst componentAst;
  private final Optional<ErrorType> interruptsExecution;

  public SimpleOperationNode(ComponentAst componentAst, ErrorHandlingContext errorHandlers,
                             ErrorTypeRepository errorTypeRepository) {
    super(getErrorHandlers(componentAst, errorHandlers, errorTypeRepository));
    this.componentAst = componentAst;
    this.interruptsExecution = getErrorTypeFromRaiseError(componentAst, errorTypeRepository);
  }

  @Override
  protected void doAccept(ChainExecutionPathTreeVisitor visitor) {
    visitor.visitSimpleOperation(this);
  }

  @Override
  public ComponentAst getComponentAst() {
    return componentAst;
  }

  @Override
  public boolean anyExecutionPathContains(Predicate<ChainExecutionPathTree> predicate) {
    return predicate.test(this) || anyErrorHandlerExecutionPathContains(predicate);
  }

  @Override
  public boolean allExecutionPathsContain(Predicate<ChainExecutionPathTree> predicate) {
    return predicate.test(this);
  }

  @Override
  public Optional<ErrorType> alwaysInterruptsWithError() {
    return interruptsExecution;
  }

  private static boolean isRaiseError(ComponentAst componentAst) {
    // For the time being, this is the only component capable of doing this
    return RAISE_ERROR_IDENTIFIER.equals(componentAst.getIdentifier());
  }

  private static ErrorType getErrorTypeFromString(String errorTypeStr, ErrorTypeRepository errorTypeRepository) {
    ComponentIdentifier errorTypeComponentIdentifier = buildFromStringRepresentation(errorTypeStr);

    return errorTypeRepository.lookupErrorType(errorTypeComponentIdentifier).orElse(new IncorrectNameErrorType());
  }

  private static Optional<ErrorType> getErrorTypeFromRaiseError(ComponentAst componentAst,
                                                                ErrorTypeRepository errorTypeRepository) {
    if (!isRaiseError(componentAst)) {
      return empty();
    }
    ComponentParameterAst typeParameter = componentAst.getParameter(DEFAULT_GROUP_NAME, "type");
    Object typeParameterValue = typeParameter.getValue().getRight();
    return typeParameterValue instanceof String ? of(getErrorTypeFromString((String) typeParameterValue, errorTypeRepository))
        : empty();
  }

  public static Optional<ChainExecutionPathTree> getErrorHandlers(ComponentAst componentAst,
                                                                  ErrorHandlingContext errorHandlers,
                                                                  ErrorTypeRepository errorTypeRepository) {
    Optional<ChainExecutionPathTree> errorHandlersFromRaiseError = getErrorTypeFromRaiseError(componentAst, errorTypeRepository)
        .flatMap(et -> errorHandlers.getMatchingErrorHandlerFor(singletonList(et)));
    if (errorHandlersFromRaiseError.isPresent()) {
      return errorHandlersFromRaiseError;
    }
    return NodeUtils.getErrorHandlers(componentAst, errorHandlers);
  }

  private static final class IncorrectNameErrorType implements ErrorType {

    @Override
    public String getIdentifier() {
      return "Incorrect";
    }

    @Override
    public String getNamespace() {
      return "Undefined";
    }

    @Override
    public ErrorType getParentErrorType() {
      return null;
    }
  }
}
