/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.cfg.internal.node.errorhandling;

import static java.util.Optional.empty;
import static java.util.Optional.of;

import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.cfg.api.ChainExecutionPathTree;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

public class ErrorHandlingContext {

  private final LinkedList<ErrorHandlerNode> context = new LinkedList<>();

  public ErrorHandlingContext(boolean addNullEH) {
    if (addNullEH) {
      context.addFirst(new ErrorHandlingExecutionPathNodeBuilder(null).build());
    }
  }

  public ErrorHandlingContext() {
    this(false);
  }

  public ErrorHandlingContext(ErrorHandlingContext other) {
    context.addAll(other.context);
  }

  public Optional<ChainExecutionPathTree> getMatchingErrorHandlerFor(List<ErrorType> errorTypes) {
    Map<ErrorType, ErrorHandlerNode> handlers = new HashMap<>();
    errorTypes.forEach(err -> context.stream().filter(handler -> handler.appliesTo(err)).findFirst()
        .ifPresent(handler -> handlers.put(err, handler)));
    return handlers.isEmpty() ? empty() : of(new ErrorHandlerExecutionChain(handlers));
  }

  public void addHandlers(List<ChainExecutionPathTree> errorHandlingNodes) {
    // Creates a copy of the current error handling context before adding any of the new handlers
    // This context will be used for any of the handlers that require propagation, or if there is an error during the error
    // handling chain
    List<ErrorHandlerNode> contextForPropagation = new LinkedList<>(context);

    for (ChainExecutionPathTree errorHandlingNode : errorHandlingNodes) {
      if (errorHandlingNode instanceof ErrorHandlerNode) {
        addHandler((ErrorHandlerNode) errorHandlingNode, contextForPropagation);
      }
    }
  }

  public void removeLastHandlers(int count) {
    for (int i = 0; i < count; i++) {
      context.removeFirst();
    }
  }

  private void addHandler(ErrorHandlerNode errorHandlingNode, List<ErrorHandlerNode> handlersForPropagation) {
    if (!errorHandlingNode.isContinueHandler()) {
      errorHandlingNode.setOuters(handlersForPropagation);
    }
    context.addFirst(errorHandlingNode);
  }

}
