/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.cfg.internal.node;

import static java.util.Collections.unmodifiableList;

import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;

import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.api.meta.model.ComponentModel;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.cfg.api.ChainExecutionPathTree;
import org.mule.runtime.cfg.api.ChainExecutionPathTreeVisitor;
import org.mule.runtime.cfg.internal.node.errorhandling.ErrorHandlerWrapperNode;
import org.mule.runtime.cfg.internal.node.errorhandling.InterruptionCapableNode;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.function.Predicate;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


public class ChainedExecutionPathNodeBuilder {

  private static final Logger LOGGER = LoggerFactory.getLogger(ChainedExecutionPathNodeBuilder.class);

  private final ComponentAst chainAst;
  private ComponentModel ownerModel;
  private ChainNode first;
  private ChainNode last;
  private final List<ChainExecutionPathTree> children = new ArrayList<>();
  private Optional<ErrorHandlerWrapperNode> ownedErrorHandler = empty();

  public ChainedExecutionPathNodeBuilder(ComponentAst chainAst) {
    this.chainAst = chainAst;
  }

  public void withOwnerModel(ComponentModel ownerModel) {
    this.ownerModel = ownerModel;
  }

  public ChainedExecutionPathNodeBuilder addChild(ChainExecutionPathTree child) {
    // Adds to the children even if the last node interrupts the execution
    children.add(child);

    // Also, we could skip adding the node to the chain if the last interrupts the execution, but in order to compute that
    // we would need to dereference chains earlier. We decided to add them to the chain and just skip them later, when visiting or
    // doing queries.
    ChainNode newNode = new ChainNode(child);
    if (first == null) {
      first = newNode;
    } else {
      last.next = of(newNode);
    }
    last = newNode;

    return this;
  }

  public ChainedExecutionPathNodeBuilder addOwnedErrorHandler(ErrorHandlerWrapperNode errorHandler) {
    this.ownedErrorHandler = of(errorHandler);
    return this;
  }

  public ChainExecutionPathTree build() {
    return new HeadNode(ownerModel, chainAst, first != null ? first : new ChainNode(NullNode.getInstance()), children,
                        ownedErrorHandler);
  }

  private static class HeadNode implements ChainExecutionPathTree, InterruptionCapableNode {

    private final ComponentModel ownerModel;
    private final ComponentAst chainAst;
    private final ChainNode firstNode;
    private final List<ChainExecutionPathTree> children;
    private final Optional<ErrorHandlerWrapperNode> errorHandlingAfterChain;
    private final LazyValue<Optional<ErrorType>> alwaysInterruptsWithError;

    public HeadNode(ComponentModel ownerModel, ComponentAst chainAst, ChainNode firstNode, List<ChainExecutionPathTree> children,
                    Optional<ErrorHandlerWrapperNode> errorHandlingAfterChain) {
      this.ownerModel = ownerModel;
      this.chainAst = chainAst;
      this.firstNode = firstNode;
      this.children = children;
      this.errorHandlingAfterChain = errorHandlingAfterChain;
      this.alwaysInterruptsWithError = new LazyValue<>(this::computeAlwaysInterruptsWithError);
      setOwningChainToChildNodes();
    }

    @Override
    public void accept(ChainExecutionPathTreeVisitor visitor) {
      this.firstNode.accept(visitor);
      this.errorHandlingAfterChain.ifPresent(errorHandler -> errorHandler.accept(visitor));
    }

    @Override
    public Optional<ComponentModel> getOwnerModel() {
      return of(ownerModel);
    }

    @Override
    public ComponentAst getComponentAst() {
      return this.chainAst;
    }

    @Override
    public boolean anyExecutionPathContains(Predicate<ChainExecutionPathTree> predicate) {
      return firstNode.anyExecutionPathContains(predicate);
    }

    @Override
    public boolean allExecutionPathsContain(Predicate<ChainExecutionPathTree> predicate) {
      return firstNode.allExecutionPathsContain(predicate);
    }

    @Override
    public List<ChainExecutionPathTree> children() {
      return unmodifiableList(this.children);
    }

    @Override
    public Optional<ErrorType> alwaysInterruptsWithError() {
      return alwaysInterruptsWithError.get();
    }

    private Optional<ErrorType> computeAlwaysInterruptsWithError() {
      Optional<ErrorType> chainError = firstNode.alwaysInterruptsWithError();
      // Checks if the error handlers from the chain would transform the error (or even remove it completely)
      if (chainError.isPresent() && errorHandlingAfterChain.isPresent()) {
        return errorHandlingAfterChain.flatMap(ehChain -> ehChain.transformsThisErrorTo(chainError.get()));
      }
      return chainError;
    }

    private void setOwningChainToChildNodes() {
      Optional<ChainNode> node = ofNullable(firstNode);
      while (node.isPresent()) {
        node.get().setOwningChain(this);
        node = node.get().next;
      }
    }
  }

  private static class ChainNode implements ChainExecutionPathTree, InterruptionCapableNode {

    private ChainExecutionPathTree owningChain;
    private final ChainExecutionPathTree delegate;
    public Optional<ChainNode> next = empty();

    private ChainNode(ChainExecutionPathTree delegate) {
      this.delegate = delegate;
    }

    public void setOwningChain(ChainExecutionPathTree owningChain) {
      this.owningChain = owningChain;
    }

    @Override
    public void accept(ChainExecutionPathTreeVisitor visitor) {
      LOGGER.debug("Visiting '{}'...", delegate.getComponentAst());

      this.delegate.accept(visitor);
      if (NodeUtils.alwaysInterruptsWithError(delegate).isPresent()) {
        visitor.chainInterruptedWithError(owningChain);
      } else {
        this.next.ifPresent(node -> node.accept(visitor));
      }
    }

    @Override
    public ComponentAst getComponentAst() {
      return this.delegate.getComponentAst();
    }

    @Override
    public boolean anyExecutionPathContains(Predicate<ChainExecutionPathTree> predicate) {
      return this.delegate.anyExecutionPathContains(predicate)
          || getNext().map(node -> node.anyExecutionPathContains(predicate)).orElse(false);
    }

    @Override
    public boolean allExecutionPathsContain(Predicate<ChainExecutionPathTree> predicate) {
      return this.delegate.allExecutionPathsContain(predicate)
          || getNext().map(node -> node.allExecutionPathsContain(predicate)).orElse(false);
    }

    @Override
    public Optional<ErrorType> alwaysInterruptsWithError() {
      Optional<ErrorType> ownErrorType = NodeUtils.alwaysInterruptsWithError(delegate);
      if (ownErrorType.isPresent()) {
        return ownErrorType;
      }
      return next.flatMap(ChainNode::alwaysInterruptsWithError);
    }

    private Optional<ChainNode> getNext() {
      return NodeUtils.alwaysInterruptsWithError(delegate).isPresent() ? empty() : next;
    }

  }

}
