/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.cfg.utils;

import static org.mule.runtime.api.component.ComponentIdentifier.buildFromStringRepresentation;
import static org.mule.runtime.api.meta.model.nested.ChainExecutionOccurrence.ONCE;
import static org.mule.runtime.api.meta.model.nested.ChainExecutionOccurrence.ONCE_OR_NONE;
import static org.mule.runtime.ast.api.error.ErrorTypeBuilder.builder;

import static java.util.Optional.of;

import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.error.ErrorTypeBuilder;
import org.mule.runtime.cfg.api.ChainExecutionPathTree;
import org.mule.runtime.cfg.api.ChainExecutionPathTreeFactory;

import java.util.Optional;

import org.mockito.stubbing.Answer;

public final class CfgTestUtils {

  public static final ComponentIdentifier SET_PAYLODAD = buildFromStringRepresentation("set-payload");
  public static final ComponentIdentifier SET_VARIABLE = buildFromStringRepresentation("set-variable");
  public static final ComponentIdentifier LOGGER = buildFromStringRepresentation("logger");
  public static final ComponentIdentifier FOREACH = buildFromStringRepresentation("foreach");
  public static final ComponentIdentifier NOOP = buildFromStringRepresentation("noop");
  public static final ComponentIdentifier NOOP2 = buildFromStringRepresentation("noop2");
  public static final ComponentIdentifier TRYSCOPE = buildFromStringRepresentation("try");
  public static final ComponentIdentifier SCATTER_GATHER = buildFromStringRepresentation("scatter-gather");
  public static final ComponentIdentifier CHOICE = buildFromStringRepresentation("choice");
  public static final ComponentIdentifier PARALLEL_FOREACH = buildFromStringRepresentation("parallel-foreach");
  public static final ComponentIdentifier DB_SELECT = ComponentIdentifier.builder().namespace("db").name("select").build();
  public static final ComponentIdentifier JMS_PUBLISH = ComponentIdentifier.builder().namespace("jms").name("publish").build();

  public static final TestCfg TEST_TREE_WITH_RAISE_ERROR = testTreeWithRaiseError();

  private CfgTestUtils() {

  }

  /**
   * Structure of this Ast: <code>
   * - Flow
   *    - Logger
   *    - Foreach
   *        - Logger
   *        - Scatter-Gather
   *            - Route
   *                - Set-payload
   *            - Route
   *                - db:select
   *        - Logger
   *    - Choice
   *        - Route
   *            - Set-payload
   *        - Route
   *            - Set-Variable
   *            - Logger
   *     - Try scope
   *         - noop
   * </code>
   */
  public static ChainExecutionPathTree testTree() {
    ComponentAst flow = MockChainBuilder.newBuilder("name")
        .addSimpleOperation(LOGGER)
        .addScope(FOREACH, ONCE_OR_NONE, fe -> fe
            .addSimpleOperation(LOGGER)
            .addRouter(SCATTER_GATHER, sg -> sg
                .addRoute(ONCE, r -> r
                    .addSimpleOperation(SET_PAYLODAD))
                .addRoute(ONCE, r -> r
                    .addSimpleOperation(DB_SELECT)))
            .addSimpleOperation(LOGGER))
        .addRouter(CHOICE, ch -> ch
            .addRoute(ONCE_OR_NONE, r -> r
                .addSimpleOperation(SET_PAYLODAD))
            .addRoute(ONCE_OR_NONE, r -> r
                .addSimpleOperation(SET_VARIABLE)
                .addSimpleOperation(LOGGER)))
        .addScope(TRYSCOPE, ONCE, tr -> tr
            .addSimpleOperation(NOOP))
        .build();

    return new ChainExecutionPathTreeFactory(getApplication()).generateFor(flow);
  }

  /**
   * Structure of this Ast: <code>
   * - Flow
   *    - Logger
   *    - Try
   *        - Logger
   *        - Choice
   *            - Route
   *                - Set-Payload
   *            - Route
   *                - db:select (errors: SOME and OTHER)
   *        - Logger
   *        - Error Handler
   *            - On error propagate (type TEST:SOME)
   *                - Set-variable
   *                - JMS Publish
   *                - Noop (no-operation)
   *            - On error continue  WITH or WITHOUT TEST:OTHER depending on the parameter
   *                - Set-variable
   *    - Error Handler
   *        - On error continue (type MULE:NOTHING)
   *           - Noop2 (another no-operation)
   *        - On error continue (no type)
   *            - set-variable
   * </code>
   */
  public static ChainExecutionPathTree testTreeWithErrorHandling(boolean continueWithAllErrors) {
    ComponentAst flow = MockChainBuilder.newBuilder("name")
        .addSimpleOperation(LOGGER)
        .addScope(TRYSCOPE, ONCE, fe -> fe
            .addSimpleOperation(LOGGER)
            .addRouter(CHOICE, sg -> sg
                .addRoute(ONCE_OR_NONE, r -> r
                    .addSimpleOperation(SET_PAYLODAD))
                .addRoute(ONCE_OR_NONE, r -> r
                    .addSimpleOperationWithErrors(DB_SELECT, "SOME", "OTHER")))
            .addSimpleOperation(LOGGER)
            .addErrorHandler(eh -> eh
                .addOnErrorPropagate("TEST:SOME", ep -> ep
                    .addSimpleOperation(SET_VARIABLE)
                    .addSimpleOperation(JMS_PUBLISH)
                    .addSimpleOperation(NOOP))
                .addOnErrorContinue(continueWithAllErrors ? null : "TEST:OTHER", ep -> ep
                    .addSimpleOperation(SET_VARIABLE))))
        .addErrorHandler(eh -> eh
            .addOnErrorContinue("MULE:NOTHING", ec -> ec
                .addSimpleOperation(NOOP2))
            .addOnErrorContinue(null, ec -> ec
                .addSimpleOperation(SET_VARIABLE)))
        .build();

    return new ChainExecutionPathTreeFactory(getApplication()).generateFor(flow);
  }

  public static TestCfg testTreeWithRaiseError() {
    return TestCfg.builder()

        .addFlow("raiseErrorAtFlowLevelNoErrorHandler", f -> f
            .addSimpleOperation(LOGGER)
            .addRaiseError("MULE:VALIDATION")
            .addSimpleOperation(LOGGER)) // Unreachable

        .addFlow("raiseErrorAtFlowLevelWithErrorHandler", f -> f
            .addSimpleOperation(LOGGER)
            .addRaiseError("MULE:VALIDATION")
            .addSimpleOperation(LOGGER) // Unreachable
            .addErrorHandler(eh -> eh
                .addOnErrorContinue(null, ec -> ec
                    .addSimpleOperation(LOGGER))))

        .addFlow("raiseErrorAtTryWithMatchingAllOnErrorContinue", f -> f
            .addSimpleOperation(LOGGER)
            .addScope(TRYSCOPE, ONCE, tr -> tr
                .addRaiseError("MULE:VALIDATION")
                .addSimpleOperation(LOGGER) // Unreachable
                .addErrorHandler(eh -> eh
                    .addOnErrorContinue(null, ec -> ec
                        .addSimpleOperation(LOGGER))))
            .addSimpleOperation(LOGGER))

        .addFlow("raiseErrorAtTryWithMatchingOnErrorContinue", f -> f
            .addSimpleOperation(LOGGER)
            .addScope(TRYSCOPE, ONCE, tr -> tr
                .addRaiseError("MULE:VALIDATION")
                .addSimpleOperation(LOGGER) // Unreachable
                .addErrorHandler(eh -> eh
                    .addOnErrorContinue("MULE:VALIDATION", ec -> ec
                        .addSimpleOperation(LOGGER))))
            .addSimpleOperation(LOGGER))

        .addFlow("raiseErrorAtTryWithMatchingOnErrorContinueWithRaiseError", f -> f
            .addSimpleOperation(LOGGER)
            .addScope(TRYSCOPE, ONCE, tr -> tr
                .addRaiseError("MULE:VALIDATION")
                .addSimpleOperation(LOGGER) // Unreachable
                .addErrorHandler(eh -> eh
                    .addOnErrorContinue("MULE:VALIDATION", ec -> ec
                        .addSimpleOperation(LOGGER)
                        .addRaiseError("MULE:NOT_PERMITTED")
                        .addSimpleOperation(LOGGER)))) // Unreachable
            .addSimpleOperation(LOGGER)) // Unreachable

        .addFlow("raiseErrorAtTryWithMatchingOnErrorContinueWithRaiseErrorThenAnotherContinue", f -> f
            .addSimpleOperation(LOGGER)
            .addScope(TRYSCOPE, ONCE, tr1 -> tr1
                .addScope(TRYSCOPE, ONCE, tr2 -> tr2
                    .addRaiseError("MULE:VALIDATION")
                    .addSimpleOperation(LOGGER) // Unreachable
                    .addErrorHandler(eh -> eh
                        .addOnErrorContinue("MULE:VALIDATION", ec -> ec
                            .addSimpleOperation(LOGGER)
                            .addRaiseError("MULE:NOT_PERMITTED")
                            .addSimpleOperation(LOGGER))))
                .addErrorHandler(eh -> eh
                    .addOnErrorContinue("MULE:NOT_PERMITTED", ec -> ec
                        .addSimpleOperation(LOGGER))))
            .addSimpleOperation(LOGGER))

        .addFlow("raiseErrorAtTryWithNotMatchingOnErrorContinue", f -> f
            .addSimpleOperation(LOGGER)
            .addScope(TRYSCOPE, ONCE, tr -> tr
                .addRaiseError("MULE:VALIDATION")
                .addSimpleOperation(LOGGER) // Unreachable
                .addErrorHandler(eh -> eh
                    .addOnErrorContinue("MULE:EXPRESSION", ec -> ec
                        .addSimpleOperation(LOGGER))))
            .addSimpleOperation(LOGGER)) // Unreachable

        .addFlow("raiseErrorAtTryWithMatchingOnErrorPropagate", f -> f
            .addSimpleOperation(LOGGER)
            .addScope(TRYSCOPE, ONCE, tr -> tr
                .addRaiseError("MULE:VALIDATION")
                .addSimpleOperation(LOGGER) // Unreachable
                .addErrorHandler(eh -> eh
                    .addOnErrorPropagate("MULE:VALIDATION", ec -> ec
                        .addSimpleOperation(LOGGER))))
            .addSimpleOperation(LOGGER)) // Unreachable

        .addFlow("raiseErrorAtChoiceAllRoutes", f -> f
            .addSimpleOperation(LOGGER)
            .addRouter(CHOICE, tr -> tr
                .addRoute(ONCE_OR_NONE, r -> r
                    .addRaiseError("MULE:VALIDATION")
                    .addSimpleOperation(LOGGER)) // Unreachable
                .addRoute(ONCE_OR_NONE, r -> r
                    .addRaiseError("MULE:VALIDATION")
                    .addSimpleOperation(LOGGER))) // Unreachable
            .addSimpleOperation(LOGGER)) // Unreachable: this is going to be a problem

        .addFlow("raiseErrorAtChoiceSomeRoutes", f -> f
            .addSimpleOperation(LOGGER)
            .addRouter(CHOICE, tr -> tr
                .addRoute(ONCE_OR_NONE, r -> r
                    .addRaiseError("MULE:VALIDATION")
                    .addSimpleOperation(LOGGER))) // Unreachable
            .addSimpleOperation(LOGGER))

        .addFlow("raiseErrorAtScatterGatherAllRoutes", f -> f
            .addSimpleOperation(LOGGER)
            .addRouter(SCATTER_GATHER, tr -> tr
                .addRoute(ONCE, r -> r
                    .addRaiseError("MULE:VALIDATION")
                    .addSimpleOperation(LOGGER)) // Unreachable
                .addRoute(ONCE, r -> r
                    .addRaiseError("MULE:VALIDATION")
                    .addSimpleOperation(LOGGER))) // Unreachable
            .addSimpleOperation(LOGGER)) // Unreachable

        .addFlow("raiseErrorAtReferencedFlowWithNoErrorHandling", f -> f
            .addFlowRef("raiseErrorAtFlowLevelNoErrorHandler")
            .addSimpleOperation(LOGGER)) // Unreachable

        .addFlow("raiseErrorAtReferencedFlowWithErrorHandling", f -> f
            .addFlowRef("raiseErrorAtFlowLevelWithErrorHandler")
            .addSimpleOperation(LOGGER))

        .addFlow("raiseErrorAtRecursiveFlow", f -> f
            .addSimpleOperation(LOGGER)
            .addScope(TRYSCOPE, ONCE, tr -> tr
                .addSimpleOperation(LOGGER)
                .addFlowRef("raiseErrorAtRecursiveFlow"))
            .addRaiseError("MULE:VALIDATION")
            .addSimpleOperation(LOGGER)) // Unreachable

        .build();
  }

  private static ArtifactAst getApplication() {
    ArtifactAst application = mock(ArtifactAst.class);
    ErrorTypeRepository repository = mock(ErrorTypeRepository.class);

    ErrorType parent = builder().identifier("ANY").namespace("MULE").build();

    when(repository.lookupErrorType(any())).thenAnswer((Answer<Optional<ErrorType>>) invocationOnMock -> {
      ComponentIdentifier identifier = invocationOnMock.getArgument(0);
      ErrorTypeBuilder builder = ErrorTypeBuilder.builder().identifier(identifier.getName()).namespace(identifier.getNamespace());
      if (!identifier.getName().equals("ANY") || !identifier.getNamespace().equals("MULE")) {
        builder.parentErrorType(parent);
      }
      return of(builder.build());
    });
    when(application.getErrorTypeRepository()).thenReturn(repository);
    return application;
  }

  public static boolean hasIdentifier(ChainExecutionPathTree tree, ComponentIdentifier identifier) {
    return tree.getComponentAst().getIdentifier().equals(identifier);
  }


}
