/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.cfg.visitors;

import static org.mule.runtime.cfg.AllureCfgConstants.CFG_FEATURE;
import static org.mule.runtime.cfg.AllureCfgConstants.Cfg.VISITORS;
import static org.mule.runtime.cfg.utils.CfgTestUtils.TEST_TREE_WITH_RAISE_ERROR;
import static org.mule.runtime.cfg.utils.CfgTestUtils.testTree;
import static org.mule.runtime.cfg.utils.CfgTestUtils.testTreeWithErrorHandling;

import static org.hamcrest.CoreMatchers.hasItems;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.collection.IsIterableContainingInOrder.contains;
import static org.hamcrest.collection.IsMapWithSize.anEmptyMap;

import org.mule.runtime.cfg.api.ChainExecutionPathTree;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.Ignore;
import org.junit.Test;

@Feature(CFG_FEATURE)
@Story(VISITORS)
public class SimpleVisitingTestCase {

  @Test
  public void countOnSimpleApplication() {
    ChainExecutionPathTree tree = testTree();
    CountingVisitor visitor = new CountingVisitor();
    tree.accept(visitor);
    assertThat(visitor.sourceCount, is(0));
    assertThat(visitor.operationCount, is(9));
    assertThat(visitor.scopeCount, is(2));
    assertThat(visitor.routerCount, is(2));
    assertThat(visitor.errorHandlerCount, is(0));
  }

  /**
   * <pre>
   * {@code
   * <flow name="name">
   *     <logger/>
   *     <try>
   *         <logger/>
   *         <choice>
   *             <when>
   *                 <set-payload/>
   *             </when>
   *             <when>
   *                 <db:select/> <!-- errors TEST:SOME and TEST:OTHER -->
   *             </when>
   *         </choice>
   *         <logger/>
   *         <error-handler>
   *             <on-error-propagate type="TEST:SOME">
   *                 <logger/>
   *                 <jms:publish/>
   *                 <noop/>
   *             </on-error-propagate>
   *             <on-error-continue>
   *                 <set-variable/>
   *             </on-error-continue>
   *         </error-handler>
   *     </try>
   *     <error-handler>
   *         <on-error-continue type="MULE:NOTHING">
   *             <noop2/>
   *         </on-error-continue>
   *         <on-error-continue>
   *             <set-variable/>
   *         </on-error-continue>
   *     </error-handler>
   * </flow>
   * }
   * </pre>
   */
  @Test
  public void countOnApplicationWithErrorHandling() {
    ChainExecutionPathTree tree = testTreeWithErrorHandling(true);
    CountingVisitor visitor = new CountingVisitor();
    tree.accept(visitor);
    assertThat(visitor.sourceCount, is(0));
    assertThat(visitor.operationCount, is(11));
    assertThat(visitor.scopeCount, is(1));
    assertThat(visitor.routerCount, is(1));
    assertThat(visitor.errorHandlerCount, is(4));
    assertThat(visitor.potentialErrorInvocations.keySet(), hasItems("name/1/3/0", "name/1/3/1", "name/2/1"));
    assertThat(visitor.potentialErrorInvocations.get("name/1/3/0"), is(1));
    assertThat(visitor.potentialErrorInvocations.get("name/1/3/1"), is(1));
    assertThat(visitor.potentialErrorInvocations.get("name/2/1"), is(1));
  }

  @Test
  public void countOnApplicationWithErrorHandlingWithNoMatchingTypes() {
    ChainExecutionPathTree tree = testTreeWithErrorHandling(false);
    CountingVisitor visitor = new CountingVisitor();
    tree.accept(visitor);
    assertThat(visitor.sourceCount, is(0));
    assertThat(visitor.operationCount, is(11));
    assertThat(visitor.scopeCount, is(1));
    assertThat(visitor.routerCount, is(1));
    assertThat(visitor.errorHandlerCount, is(4));
    assertThat(visitor.potentialErrorInvocations.keySet(), hasItems("name/1/3/0", "name/1/3/1", "name/2/1"));
    assertThat(visitor.potentialErrorInvocations.get("name/1/3/0"), is(1));
    assertThat(visitor.potentialErrorInvocations.get("name/1/3/1"), is(1));
    assertThat(visitor.potentialErrorInvocations.get("name/2/1"), is(1));
  }

  /**
   * <pre>
   * {@code
   * <flow name="raiseErrorAtFlowLevelNoErrorHandler">
   *   <logger/>
   *   <raise-error type="MULE:VALIDATION"/>
   *   <logger/> <!-- unreachable -->
   * </flow>
   * 
  </pre>
   */
  @Test
  public void countOnApplicationWithRaiseErrorAtFlowLevelNoErrorHandler() {
    String flowName = "raiseErrorAtFlowLevelNoErrorHandler";
    ChainExecutionPathTree tree = TEST_TREE_WITH_RAISE_ERROR.getCfgForFlow(flowName);
    CountingVisitor visitor = new CountingVisitor();
    tree.accept(visitor);
    assertThat(visitor.sourceCount, is(0));
    assertThat(visitor.operationCount, is(2));
    assertThat(visitor.scopeCount, is(0));
    assertThat(visitor.routerCount, is(0));
    assertThat(visitor.errorHandlerCount, is(0));
    assertThat(visitor.potentialErrorInvocations, is(anEmptyMap()));
    assertThat(visitor.interruptedChains, contains(flowName));
  }

  /**
   * <pre>
   * {@code
   * <flow name="raiseErrorAtFlowLevelWithErrorHandler">
   *   <logger/>
   *   <raise-error type="MULE:VALIDATION"/>
   *   <logger/> <!-- unreachable -->
   *   <error-handler>
   *       <on-error-continue>
   *           <logger/>
   *       </on-error-continue>
   *   </error-handler>
   * </flow>
   * }
   * </pre>
   */
  @Test
  public void countOnApplicationWithRaiseErrorAtFlowLevelWithErrorHandler() {
    String flowName = "raiseErrorAtFlowLevelWithErrorHandler";
    ChainExecutionPathTree tree = TEST_TREE_WITH_RAISE_ERROR.getCfgForFlow(flowName);
    CountingVisitor visitor = new CountingVisitor();
    tree.accept(visitor);
    assertThat(visitor.sourceCount, is(0));
    assertThat(visitor.operationCount, is(3));
    assertThat(visitor.scopeCount, is(0));
    assertThat(visitor.routerCount, is(0));
    assertThat(visitor.errorHandlerCount, is(1));
    assertThat(visitor.potentialErrorInvocations.keySet(), hasItems(flowName + "/3/0"));
    assertThat(visitor.potentialErrorInvocations.get(flowName + "/3/0"), is(1));
    assertThat(visitor.interruptedChains, contains(flowName));
  }

  /**
   * <pre>
   * {@code
   * <flow name="raiseErrorAtTryWithMatchingAllOnErrorContinue">
   *     <logger/>
   *     <try>
   *         <raise-error type="MULE:VALIDATION"/>
   *         <logger/> <!-- unreachable -->
   *         <error-handler>
   *             <on-error-continue>
   *                 <logger/>
   *             </on-error-continue>
   *         </error-handler>
   *     </try>
   *     <logger/>
   * </flow>
   * }
   * </pre>
   */
  @Test
  public void countOnApplicationWithRaiseErrorAtTryWithMatchingAllOnErrorContinue() {
    String flowName = "raiseErrorAtTryWithMatchingAllOnErrorContinue";
    ChainExecutionPathTree tree = TEST_TREE_WITH_RAISE_ERROR.getCfgForFlow(flowName);
    CountingVisitor visitor = new CountingVisitor();
    tree.accept(visitor);
    assertThat(visitor.sourceCount, is(0));
    assertThat(visitor.operationCount, is(4));
    assertThat(visitor.scopeCount, is(1));
    assertThat(visitor.routerCount, is(0));
    assertThat(visitor.errorHandlerCount, is(1));
    assertThat(visitor.potentialErrorInvocations.keySet(), hasItems(flowName + "/1/2/0"));
    assertThat(visitor.potentialErrorInvocations.get(flowName + "/1/2/0"), is(1));
    assertThat(visitor.interruptedChains, contains(flowName + "/1"));
  }

  /**
   * <pre>
   * {@code
   * <flow name="raiseErrorAtTryWithMatchingOnErrorContinue">
   *     <logger/>
   *     <try>
   *         <raise-error type="MULE:VALIDATION"/>
   *         <logger/> <!-- unreachable -->
   *         <error-handler>
   *             <on-error-continue type="MULE:VALIDATION">
   *                 <logger/>
   *             </on-error-continue>
   *         </error-handler>
   *     </try>
   *     <logger/>
   * </flow>
   * }
   * </pre>
   */
  @Test
  public void countOnApplicationWithRaiseErrorAtTryWithMatchingOnErrorContinue() {
    String flowName = "raiseErrorAtTryWithMatchingOnErrorContinue";
    ChainExecutionPathTree tree = TEST_TREE_WITH_RAISE_ERROR.getCfgForFlow(flowName);
    CountingVisitor visitor = new CountingVisitor();
    tree.accept(visitor);
    assertThat(visitor.sourceCount, is(0));
    assertThat(visitor.operationCount, is(4));
    assertThat(visitor.scopeCount, is(1));
    assertThat(visitor.routerCount, is(0));
    assertThat(visitor.errorHandlerCount, is(1));
    assertThat(visitor.potentialErrorInvocations.keySet(), hasItems(flowName + "/1/2/0"));
    assertThat(visitor.potentialErrorInvocations.get(flowName + "/1/2/0"), is(1));
    assertThat(visitor.interruptedChains, contains(flowName + "/1"));
  }

  /**
   * <pre>
   * {@code
   * <flow name="raiseErrorAtTryWithMatchingOnErrorContinueWithRaiseError">
   *     <logger/>
   *     <try>
   *         <raise-error type="MULE:VALIDATION"/>
   *         <logger/> <!-- unreachable -->
   *         <error-handler>
   *             <on-error-continue type="MULE:VALIDATION">
   *                 <logger/>
   *                 <raise-error type="MULE:NOT_PERMITTED"/>
   *                 <logger/>  <!-- unreachable -->
   *             </on-error-continue>
   *         </error-handler>
   *     </try>
   *     <logger/>  <!-- unreachable -->
   * </flow>
   * }
   * </pre>
   */
  @Test
  public void countOnApplicationWithRaiseErrorAtTryWithMatchingOnErrorContinueWithRaiseError() {
    String flowName = "raiseErrorAtTryWithMatchingOnErrorContinueWithRaiseError";
    ChainExecutionPathTree tree = TEST_TREE_WITH_RAISE_ERROR.getCfgForFlow(flowName);
    CountingVisitor visitor = new CountingVisitor();
    tree.accept(visitor);
    assertThat(visitor.sourceCount, is(0));
    assertThat(visitor.operationCount, is(4));
    assertThat(visitor.scopeCount, is(1));
    assertThat(visitor.routerCount, is(0));
    assertThat(visitor.errorHandlerCount, is(1));
    assertThat(visitor.potentialErrorInvocations.keySet(), hasItems(flowName + "/1/2/0"));
    assertThat(visitor.potentialErrorInvocations.get(flowName + "/1/2/0"), is(1));
    assertThat(visitor.interruptedChains, contains(flowName + "/1", flowName + "/1/2/0", flowName));
  }

  /**
   * <pre>
   * {@code
   * <flow name="raiseErrorAtTryWithMatchingOnErrorContinueWithRaiseErrorThenAnotherContinue">
   *     <logger/>
   *     <try>
   *         <try>
   *             <raise-error type="MULE:VALIDATION"/>
   *             <logger/> <!-- unreachable -->
   *             <error-handler>
   *                 <on-error-continue type="MULE:VALIDATION">
   *                     <logger/>
   *                     <raise-error type="MULE:NOT_PERMITTED"/>
   *                     <logger/>  <!-- unreachable -->
   *                 </on-error-continue>
   *             </error-handler>
   *         </try>
   *         <error-handler>
   *             <on-error-continue type="MULE:NOT_PERMITTED">
   *                 <logger/>
   *             </on-error-continue>
   *         </error-handler>
   *     </try>
   *     <logger/>
   * </flow>
   * }
   * </pre>
   */
  @Test
  public void countOnApplicationWithRaiseErrorAtTryWithMatchingOnErrorContinueWithRaiseErrorThenAnotherContinue() {
    String flowName = "raiseErrorAtTryWithMatchingOnErrorContinueWithRaiseErrorThenAnotherContinue";
    ChainExecutionPathTree tree = TEST_TREE_WITH_RAISE_ERROR.getCfgForFlow(flowName);
    CountingVisitor visitor = new CountingVisitor();
    tree.accept(visitor);
    assertThat(visitor.sourceCount, is(0));
    assertThat(visitor.operationCount, is(6));
    assertThat(visitor.scopeCount, is(2));
    assertThat(visitor.routerCount, is(0));
    assertThat(visitor.errorHandlerCount, is(2));
    assertThat(visitor.potentialErrorInvocations.keySet(), hasItems(flowName + "/1/0/2/0", flowName + "/1/1/0"));
    assertThat(visitor.potentialErrorInvocations.get(flowName + "/1/0/2/0"), is(1));
    assertThat(visitor.potentialErrorInvocations.get(flowName + "/1/1/0"), is(1));
    assertThat(visitor.interruptedChains, contains(flowName + "/1/0", flowName + "/1/0/2/0", flowName + "/1"));
  }

  /**
   * <pre>
   * {@code
   * <flow name="raiseErrorAtTryWithNotMatchingOnErrorContinue">
   *     <logger/>
   *     <try>
   *         <raise-error type="MULE:VALIDATION"/>
   *         <logger/> <!-- unreachable -->
   *         <error-handler>
   *             <on-error-continue type="MULE:EXPRESSION">
   *                 <logger/>
   *             </on-error-continue>
   *         </error-handler>
   *     </try>
   *     <logger/> <!-- unreachable -->
   * </flow>
   * }
   * </pre>
   */
  @Test
  public void countOnApplicationWithRaiseErrorAtTryWithNotMatchingOnErrorContinue() {
    String flowName = "raiseErrorAtTryWithNotMatchingOnErrorContinue";
    ChainExecutionPathTree tree = TEST_TREE_WITH_RAISE_ERROR.getCfgForFlow(flowName);
    CountingVisitor visitor = new CountingVisitor();
    tree.accept(visitor);
    assertThat(visitor.sourceCount, is(0));
    assertThat(visitor.operationCount, is(3));
    assertThat(visitor.scopeCount, is(1));
    assertThat(visitor.routerCount, is(0));
    assertThat(visitor.errorHandlerCount, is(1));
    assertThat(visitor.potentialErrorInvocations, is(anEmptyMap()));
    assertThat(visitor.interruptedChains, contains(flowName + "/1", flowName));
  }

  /**
   * <pre>
   * {@code
   * <flow name="raiseErrorAtTryWithMatchingOnErrorPropagate">
   *     <logger/>
   *     <try>
   *         <raise-error type="MULE:VALIDATION"/>
   *         <logger/> <!-- unreachable -->
   *         <error-handler>
   *             <on-error-propagate>
   *                 <logger/>
   *             </on-error-propagate>
   *         </error-handler>
   *     </try>
   *     <logger/> <!-- unreachable -->
   * </flow>
   * }
   * </pre>
   */
  @Test
  public void countOnApplicationWithRaiseErrorAtTryWithMatchingOnErrorPropagate() {
    String flowName = "raiseErrorAtTryWithMatchingOnErrorPropagate";
    ChainExecutionPathTree tree = TEST_TREE_WITH_RAISE_ERROR.getCfgForFlow(flowName);
    CountingVisitor visitor = new CountingVisitor();
    tree.accept(visitor);
    assertThat(visitor.sourceCount, is(0));
    assertThat(visitor.operationCount, is(3));
    assertThat(visitor.scopeCount, is(1));
    assertThat(visitor.routerCount, is(0));
    assertThat(visitor.errorHandlerCount, is(1));
    assertThat(visitor.potentialErrorInvocations.keySet(), hasItems(flowName + "/1/2/0"));
    assertThat(visitor.potentialErrorInvocations.get(flowName + "/1/2/0"), is(1));
    assertThat(visitor.interruptedChains, contains(flowName + "/1", flowName));
  }

  /**
   * <pre>
   * {@code
   * <flow name="raiseErrorAtChoiceAllRoutes">
   *     <logger/>
   *     <choice>
   *         <when>
   *             <raise-error type="MULE:VALIDATION"/>
   *             <logger/> <!-- unreachable -->
   *         </when>
   *         <otherwise>
   *             <raise-error type="MULE:VALIDATION"/>
   *             <logger/> <!-- unreachable -->
   *         </otherwise>
   *     </choice>
   *     <logger/> <!-- unreachable: this is going to be a problem -->
   * </flow>
   * }
   * </pre>
   */
  @Test
  @Ignore("W-17130449: Choice is a special case because all routes are ONCE_OR_NONE but in the presence of otherwise, one will execute")
  public void countOnApplicationWithRaiseErrorAtChoiceAllRoutes() {
    String flowName = "raiseErrorAtChoiceAllRoutes";
    ChainExecutionPathTree tree = TEST_TREE_WITH_RAISE_ERROR.getCfgForFlow(flowName);
    CountingVisitor visitor = new CountingVisitor();
    tree.accept(visitor);
    assertThat(visitor.sourceCount, is(0));
    assertThat(visitor.operationCount, is(3));
    assertThat(visitor.scopeCount, is(0));
    assertThat(visitor.routerCount, is(1));
    assertThat(visitor.errorHandlerCount, is(0));
    assertThat(visitor.potentialErrorInvocations, is(anEmptyMap()));
    assertThat(visitor.interruptedChains, contains(flowName + "/1/0", flowName + "/1/1"));
  }

  /**
   * <pre>
   * {@code
   * <flow name="raiseErrorAtChoiceSomeRoutes">
   *     <logger/>
   *     <choice>
   *         <when>
   *             <raise-error type="MULE:VALIDATION"/>
   *             <logger/> <!-- unreachable -->
   *         </when>
   *     </choice>
   *     <logger/>
   * </flow>
   * }
   * </pre>
   */
  @Test
  public void countOnApplicationWithRaiseErrorAtChoiceSomeRoutes() {
    String flowName = "raiseErrorAtChoiceSomeRoutes";
    ChainExecutionPathTree tree = TEST_TREE_WITH_RAISE_ERROR.getCfgForFlow(flowName);
    CountingVisitor visitor = new CountingVisitor();
    tree.accept(visitor);
    assertThat(visitor.sourceCount, is(0));
    assertThat(visitor.operationCount, is(3));
    assertThat(visitor.scopeCount, is(0));
    assertThat(visitor.routerCount, is(1));
    assertThat(visitor.errorHandlerCount, is(0));
    assertThat(visitor.potentialErrorInvocations, is(anEmptyMap()));
    assertThat(visitor.interruptedChains, contains(flowName + "/1/0"));
  }

  /**
   * <pre>
   * {@code
   * <flow name="raiseErrorAtScatterGatherAllRoutes">
   *     <logger/>
   *     <scatter-gather>
   *         <route>
   *             <raise-error type="MULE:VALIDATION"/>
   *             <logger/> <!-- unreachable -->
   *         </route>
   *         <route>
   *             <raise-error type="MULE:VALIDATION"/>
   *             <logger/> <!-- unreachable -->
   *         </route>
   *     </scatter-gather>
   *     <logger/> <!-- unreachable -->
   * </flow>
   * }
   * </pre>
   */
  @Test
  public void countOnApplicationWithRaiseErrorAtScatterGatherAllRoutes() {
    String flowName = "raiseErrorAtScatterGatherAllRoutes";
    ChainExecutionPathTree tree = TEST_TREE_WITH_RAISE_ERROR.getCfgForFlow(flowName);
    CountingVisitor visitor = new CountingVisitor();
    tree.accept(visitor);
    assertThat(visitor.sourceCount, is(0));
    assertThat(visitor.operationCount, is(3));
    assertThat(visitor.scopeCount, is(0));
    assertThat(visitor.routerCount, is(1));
    assertThat(visitor.errorHandlerCount, is(0));
    assertThat(visitor.potentialErrorInvocations, is(anEmptyMap()));
    assertThat(visitor.interruptedChains, contains(flowName + "/1/0", flowName + "/1/1", flowName));
  }

  /**
   * <pre>
   * {@code
   * <flow name="raiseErrorAtReferencedFlowWithNoErrorHandling">
   *     <flow-ref name="raiseErrorAtFlowLevelNoErrorHandler"/> <!-- has two operations -->
   *     <logger/> <!-- unreachable -->
   * </flow>
   * }
   * </pre>
   */
  @Test
  public void countOnApplicationWithRaiseErrorAtReferencedFlowWithNoErrorHandling() {
    String flowName = "raiseErrorAtReferencedFlowWithNoErrorHandling";
    String referencedFlowName = "raiseErrorAtFlowLevelNoErrorHandler";
    ChainExecutionPathTree tree = TEST_TREE_WITH_RAISE_ERROR.getCfgForFlow(flowName);
    CountingVisitor visitor = new CountingVisitor();
    tree.accept(visitor);
    assertThat(visitor.sourceCount, is(0));
    assertThat(visitor.operationCount, is(2));
    assertThat(visitor.scopeCount, is(0));
    assertThat(visitor.routerCount, is(0));
    assertThat(visitor.errorHandlerCount, is(0));
    assertThat(visitor.potentialErrorInvocations, is(anEmptyMap()));
    assertThat(visitor.interruptedChains, contains(referencedFlowName, flowName));
  }

  /**
   * <pre>
   * {@code
   * <flow name="raiseErrorAtReferencedFlowWithErrorHandling">
   *     <flow-ref name="raiseErrorAtFlowLevelWithErrorHandler"/> <!-- has three operations -->
   *     <logger/>
   * </flow>
   * }
   * </pre>
   */
  @Test
  public void countOnApplicationWithRaiseErrorAtReferencedFlowWithErrorHandling() {
    String flowName = "raiseErrorAtReferencedFlowWithErrorHandling";
    String referencedFlowName = "raiseErrorAtFlowLevelWithErrorHandler";
    ChainExecutionPathTree tree = TEST_TREE_WITH_RAISE_ERROR.getCfgForFlow(flowName);
    CountingVisitor visitor = new CountingVisitor();
    tree.accept(visitor);
    assertThat(visitor.sourceCount, is(0));
    assertThat(visitor.operationCount, is(4));
    assertThat(visitor.scopeCount, is(0));
    assertThat(visitor.routerCount, is(0));
    assertThat(visitor.errorHandlerCount, is(1));
    assertThat(visitor.potentialErrorInvocations.keySet(), hasItems(referencedFlowName + "/3/0"));
    assertThat(visitor.potentialErrorInvocations.get(referencedFlowName + "/3/0"), is(1));
    assertThat(visitor.interruptedChains, contains(referencedFlowName));
  }

  /**
   * <pre>
   * {@code
   * <flow name="raiseErrorAtRecursiveFlow">
   *     <logger/>
   *     <try>
   *         <logger/>
   *         <flow-ref name="raiseErrorAtRecursiveFlow"/>
   *     </try>
   *     <raise-error type="MULE:VALIDATION"/>
   *     <logger/> <!-- unreachable -->
   * </flow>
   * }
   * </pre>
   */
  @Test
  public void countOnApplicationWithRaiseErrorAtRecursiveFlow() {
    ChainExecutionPathTree tree = TEST_TREE_WITH_RAISE_ERROR.getCfgForFlow("raiseErrorAtRecursiveFlow");
    CountingVisitor visitor = new CountingVisitor();
    tree.accept(visitor);
    assertThat(visitor.sourceCount, is(0));
    assertThat(visitor.operationCount, is(6)); // It goes to each operation twice because of the recursion and then stops
    assertThat(visitor.scopeCount, is(2)); // Same with the scopes
    assertThat(visitor.routerCount, is(0));
    assertThat(visitor.errorHandlerCount, is(0));
    assertThat(visitor.potentialErrorInvocations, is(anEmptyMap()));
  }

}
