/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.cfg.visitors;

import static org.mule.runtime.cfg.AllureCfgConstants.CFG_FEATURE;
import static org.mule.runtime.cfg.AllureCfgConstants.Cfg.VISITORS;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.hasSize;
import static java.util.Arrays.asList;
import static java.util.stream.Collectors.toList;
import static org.mule.runtime.cfg.utils.CfgTestUtils.*;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.hamcrest.BaseMatcher;
import org.hamcrest.Description;
import org.junit.Test;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.cfg.api.ChainExecutionPathTree;

import java.util.List;

@Feature(CFG_FEATURE)
@Story(VISITORS)
public class AllPathsTestCase {

  private static final String LOGGER_CMP = "logger";
  private static final String FOREACH_CMP = "foreach";
  private static final String SCATTER_GATHER_CMP = "scatter-gather";
  private static final String SELECT_CMP = "db:select";
  private static final String CHOICE_CMP = "choice";
  private static final String SET_VARIABLE_CMP = "set-variable";
  private static final String SET_PAYLOAD_CMP = "set-payload";
  private static final String TRY_CMP = "try";
  private static final String NOOP_CMP = "noop";

  @Test
  public void visitOnSimpleApplication() {
    ChainExecutionPathTree tree = testTree();
    AllExecutionPathsVisitor visitor = new AllExecutionPathsVisitor();
    tree.accept(visitor);
    List<List<ComponentAst>> possibleExecutions = visitor.getAllPossibleExecutions();
    assertThat(possibleExecutions, hasSize(4));
    List<List<String>> identifiers = possibleExecutions.stream()
        .map(l -> l.stream().map(comp -> comp.getIdentifier().toString()).collect(toList())).collect(toList());
    assertThat(identifiers, new AnySubCollectionIs(LOGGER_CMP, FOREACH_CMP, LOGGER_CMP, SCATTER_GATHER_CMP, SELECT_CMP,
                                                   LOGGER_CMP, CHOICE_CMP, SET_VARIABLE_CMP, LOGGER_CMP, TRY_CMP, NOOP_CMP));
    assertThat(identifiers, new AnySubCollectionIs(LOGGER_CMP, FOREACH_CMP, LOGGER_CMP, SCATTER_GATHER_CMP, SELECT_CMP,
                                                   LOGGER_CMP, CHOICE_CMP, SET_PAYLOAD_CMP, TRY_CMP, NOOP_CMP));
    assertThat(identifiers, new AnySubCollectionIs(LOGGER_CMP, FOREACH_CMP, LOGGER_CMP, SCATTER_GATHER_CMP, SET_PAYLOAD_CMP,
                                                   LOGGER_CMP, CHOICE_CMP, SET_VARIABLE_CMP, LOGGER_CMP, TRY_CMP, NOOP_CMP));
    assertThat(identifiers, new AnySubCollectionIs(LOGGER_CMP, FOREACH_CMP, LOGGER_CMP, SCATTER_GATHER_CMP, SET_PAYLOAD_CMP,
                                                   LOGGER_CMP, CHOICE_CMP, SET_PAYLOAD_CMP, TRY_CMP, NOOP_CMP));
  }

  private static final class AnySubCollectionIs extends BaseMatcher<List<List<String>>> {

    private final List<String> expected;

    public AnySubCollectionIs(String... expected) {
      this.expected = asList(expected);
    }

    @Override
    public boolean matches(Object o) {
      List<List<String>> actual = (List<List<String>>) o;
      return actual.stream().anyMatch(sublist -> sublist.equals(expected));
    }

    @Override
    public void describeTo(Description description) {
      description.appendText("None of the sub collections matches the list " + expected.toString());
    }
  }
}
