/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.cfg.utils;

import static java.util.Arrays.asList;

import static org.mule.runtime.api.component.ComponentIdentifier.buildFromStringRepresentation;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.ERROR_HANDLER;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.FLOW;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.ON_ERROR;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.OPERATION;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.ROUTE;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.ROUTER;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.SCOPE;
import static org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED;
import static org.mule.runtime.api.meta.ExpressionSupport.SUPPORTED;
import static org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.DEFAULT_GROUP_NAME;
import static org.mule.runtime.ast.api.ComponentMetadataAst.EMPTY_METADATA;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.Collections.singletonList;

import static java.util.Optional.empty;
import static java.util.Optional.of;

import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.metadata.api.ClassTypeLoader;
import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.TypedComponentIdentifier;
import org.mule.runtime.api.meta.ExpressionSupport;
import org.mule.runtime.api.meta.MuleVersion;
import org.mule.runtime.api.meta.model.ComponentModelVisitor;
import org.mule.runtime.api.meta.model.ComponentVisibility;
import org.mule.runtime.api.meta.model.ModelProperty;
import org.mule.runtime.api.meta.model.deprecated.DeprecationModel;
import org.mule.runtime.api.meta.model.display.DisplayModel;
import org.mule.runtime.api.meta.model.error.ErrorModel;
import org.mule.runtime.api.meta.model.error.ImmutableErrorModel;
import org.mule.runtime.api.meta.model.nested.ChainExecutionOccurrence;
import org.mule.runtime.api.meta.model.nested.HasChainExecutionOccurrence;
import org.mule.runtime.api.meta.model.nested.NestableElementModel;
import org.mule.runtime.api.meta.model.nested.NestableElementModelVisitor;
import org.mule.runtime.api.meta.model.nested.NestedChainModel;
import org.mule.runtime.api.meta.model.nested.NestedRouteModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.meta.model.stereotype.StereotypeModel;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentGenerationInformation;
import org.mule.runtime.ast.api.ComponentMetadataAst;
import org.mule.runtime.ast.api.builder.ComponentAstBuilder;
import org.mule.runtime.ast.internal.DefaultComponentParameterAst;
import org.mule.runtime.ast.internal.builder.BaseComponentAstBuilder;
import org.mule.runtime.ast.internal.builder.ComponentLocationVisitor;
import org.mule.runtime.ast.internal.builder.DefaultComponentAstBuilder;
import org.mule.runtime.ast.internal.builder.PropertiesResolver;
import org.mule.runtime.ast.internal.model.DefaultExtensionModelHelper;
import org.mule.runtime.ast.internal.model.ParameterModelUtils;
import org.mule.runtime.extension.api.declaration.type.ExtensionsTypeLoaderFactory;
import org.mule.sdk.api.stereotype.MuleStereotypes;

import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;

public class MockChainBuilder {

  private static final ClassTypeLoader TYPE_LOADER = ExtensionsTypeLoaderFactory.getDefault()
      .createTypeLoader(MockChainBuilder.class.getClassLoader());
  private static final MetadataType STRING_TYPE = TYPE_LOADER.load(String.class);

  private static final ComponentIdentifier FLOW_COMP = buildFromStringRepresentation("flow");
  private static final ComponentIdentifier FLOW_REF = buildFromStringRepresentation("flow-ref");
  private static final ComponentIdentifier ROUTE_ELEMENT = buildFromStringRepresentation("route");
  private static final ComponentIdentifier LIST_OF_ROUTES_ELEMENT = buildFromStringRepresentation("list-of-routes");
  private static final ComponentIdentifier ERROR_HANDLER_COMPONENT = buildFromStringRepresentation("error-handler");
  private static final ComponentIdentifier ERROR_CONTINUE = buildFromStringRepresentation("on-error-continue");
  private static final ComponentIdentifier ERROR_PROPAGATE = buildFromStringRepresentation("on-error-propagate");
  public static final ComponentIdentifier RAISE_ERROR = buildFromStringRepresentation("raise-error");

  private final ComponentAstBuilder delegate;

  public static MockChainBuilder newBuilder(String flowName) {
    return new MockChainBuilder(getFlowBuilder(flowName));
  }

  private MockChainBuilder(ComponentAstBuilder delegate) {
    this.delegate = delegate;
  }

  private static ComponentAstBuilder getFlowBuilder(String flowName) {
    DefaultComponentAstBuilder flowBuilder =
        new DefaultComponentAstBuilder(new PropertiesResolver(),
                                       mock(DefaultExtensionModelHelper.class),
                                       emptyList(),
                                       0,
                                       new ComponentLocationVisitor(),
                                       new ParameterModelUtils());
    flowBuilder.withIdentifier(FLOW_COMP)
        .withRawParameter("name", flowName)
        .withMetadata(EMPTY_METADATA);
    addStringParameter(flowBuilder, "flowName", flowName, true, NOT_SUPPORTED, emptyList());
    flowBuilder.withComponentType(FLOW);
    return flowBuilder;
  }

  public MockChainBuilder addSimpleOperation(ComponentIdentifier componentIdentifier) {
    addChildComponentBuilder(componentIdentifier, OPERATION);
    return this;
  }

  public MockChainBuilder addSimpleOperationWithErrors(ComponentIdentifier componentIdentifier, String... errors) {
    DefaultComponentAstBuilder componentAstBuilder = addChildComponentBuilder(componentIdentifier, OPERATION);
    addErrors(componentAstBuilder, errors);
    return this;
  }

  public MockChainBuilder addRaiseError(String errorType) {
    DefaultComponentAstBuilder builder = addChildComponentBuilder(RAISE_ERROR, OPERATION);
    addStringParameter(builder, "type", errorType, false, NOT_SUPPORTED, emptyList());
    return this;
  }

  public MockChainBuilder addFlowRef(String name) {
    DefaultComponentAstBuilder builder = addChildComponentBuilder(FLOW_REF, OPERATION);
    addStringParameter(builder, "name", name, false, SUPPORTED, asList(MuleStereotypes.FLOW, MuleStereotypes.SUB_FLOW));
    return this;
  }

  public MockChainBuilder addScope(ComponentIdentifier componentIdentifier,
                                   ChainExecutionOccurrence chainExecutionOccurrence,
                                   Consumer<MockChainBuilder> chainConfigurer) {
    final OperationModel scopeOperationModel = mock(OperationModel.class);
    final List nestedChains = asList(new MockChainModel(chainExecutionOccurrence));
    when(scopeOperationModel.getNestedComponents()).thenReturn(nestedChains);

    ComponentAstBuilder builder = addChildComponentBuilder(componentIdentifier, SCOPE)
        .withComponentModel(scopeOperationModel);

    chainConfigurer.accept(new MockChainBuilder(builder));
    return this;
  }

  public MockChainBuilder addRouter(ComponentIdentifier componentIdentifier, Consumer<MockChainBuilder> routerConfigurer) {
    ComponentAstBuilder builder = addChildComponentBuilder(componentIdentifier, ROUTER)
        .withComponentModel(mock(OperationModel.class));

    routerConfigurer.accept(new MockChainBuilder(builder));
    return this;
  }

  public MockChainBuilder addRoute(ChainExecutionOccurrence chainExecutionOccurrence,
                                   Consumer<MockChainBuilder> chainConfigurer) {
    ComponentAstBuilder builder = addChildComponentBuilder(ROUTE_ELEMENT, ROUTE)
        .withComponentModel(new MockRouteModel(chainExecutionOccurrence));

    chainConfigurer.accept(new MockChainBuilder(builder));
    return this;
  }

  public MockChainBuilder addListOfRoutes(ChainExecutionOccurrence chainExecutionOccurrence,
                                          int routes,
                                          Consumer<MockChainBuilder> chainConfigurer) {
    ComponentAstBuilder wrapperBuilder = addChildComponentBuilder(LIST_OF_ROUTES_ELEMENT, ROUTE);

    for (int i = 0; i < routes; ++i) {
      ComponentAstBuilder builder = wrapperBuilder.addChildComponent()
          .withIdentifier(ROUTE_ELEMENT);
      builder = ((BaseComponentAstBuilder) builder).withComponentType(ROUTE);
      builder = ((BaseComponentAstBuilder) builder).withComponentModel(new MockRouteModel(chainExecutionOccurrence));
      builder = ((BaseComponentAstBuilder) builder).withMetadata(mock(ComponentMetadataAst.class));

      chainConfigurer.accept(new MockChainBuilder(builder));
    }

    return this;
  }

  public MockChainBuilder addErrorHandler(Consumer<MockChainBuilder> chainConfigurer) {
    DefaultComponentAstBuilder errorHandler = addChildComponentBuilder(ERROR_HANDLER_COMPONENT, ERROR_HANDLER);
    addStringParameter(errorHandler, "ref", null, true, NOT_SUPPORTED, singletonList(MuleStereotypes.ERROR_HANDLER));

    chainConfigurer.accept(new MockChainBuilder(errorHandler));
    return this;
  }

  public MockChainBuilder addOnErrorPropagate(String type, Consumer<MockChainBuilder> chainConfigurer) {
    return addOnError(ERROR_PROPAGATE, type, chainConfigurer);
  }

  public MockChainBuilder addOnErrorContinue(String type, Consumer<MockChainBuilder> chainConfigurer) {
    return addOnError(ERROR_CONTINUE, type, chainConfigurer);
  }

  private MockChainBuilder addOnError(ComponentIdentifier componentIdentifier,
                                      String type,
                                      Consumer<MockChainBuilder> chainConfigurer) {
    DefaultComponentAstBuilder propagateEH = addChildComponentBuilder(componentIdentifier, ON_ERROR);;
    addStringParameter(propagateEH, "type", type, false, NOT_SUPPORTED, emptyList());

    chainConfigurer.accept(new MockChainBuilder(propagateEH));
    return this;
  }

  public ComponentAst build() {
    return delegate.build();
  }

  private DefaultComponentAstBuilder addChildComponentBuilder(ComponentIdentifier identifier,
                                                              TypedComponentIdentifier.ComponentType componentType) {
    DefaultComponentAstBuilder componentBuilder = (DefaultComponentAstBuilder) delegate.addChildComponent()
        .withIdentifier(identifier)
        .withMetadata(EMPTY_METADATA);
    componentBuilder.withComponentType(componentType);
    return componentBuilder;
  }

  private static ParameterGroupModel createMockParameterGroup(ParameterModel parameterModel) {
    ParameterGroupModel groupModel = mock(ParameterGroupModel.class);
    when(groupModel.getName()).thenReturn(DEFAULT_GROUP_NAME);
    when(groupModel.getParameterModels()).thenReturn(emptyList());
    when(groupModel.getParameterModels()).thenReturn(singletonList(parameterModel));
    when(groupModel.getParameter(parameterModel.getName())).thenReturn(of(parameterModel));
    return groupModel;
  }

  private static void addErrors(DefaultComponentAstBuilder builder, String... errors) {
    OperationModel model = mock(OperationModel.class);
    Set<ErrorModel> errorsModels = new HashSet<>();
    ErrorModel universalParent = new ImmutableErrorModel("ANY", "MULE", true, null);
    for (String err : errors) {
      errorsModels.add(new ImmutableErrorModel(err, "TEST", true, universalParent));
    }
    when(model.getErrorModels()).thenReturn(errorsModels);
    builder.withComponentModel(model);
  }

  private static void addStringParameter(DefaultComponentAstBuilder componentBuilder,
                                         String paramName,
                                         String paramValue,
                                         boolean isComponentId,
                                         ExpressionSupport expressionSupport,
                                         List<StereotypeModel> allowedStereotypes) {
    ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getName()).thenReturn(paramName);
    when(parameterModel.getType()).thenReturn(STRING_TYPE);
    when(parameterModel.isComponentId()).thenReturn(isComponentId);
    when(parameterModel.getExpressionSupport()).thenReturn(expressionSupport);
    when(parameterModel.getAllowedStereotypes()).thenReturn(allowedStereotypes);

    ParameterGroupModel defaultGroup = createMockParameterGroup(parameterModel);
    if (paramValue != null) {
      componentBuilder.withParameter(parameterModel, defaultGroup,
                                     new DefaultComponentParameterAst(paramValue,
                                                                      parameterModel,
                                                                      defaultGroup,
                                                                      mock(ComponentGenerationInformation.class),
                                                                      new PropertiesResolver(),
                                                                      new ParameterModelUtils()),
                                     empty());
    }

    ParameterizedModel parameterized = mock(ParameterizedModel.class);
    when(parameterized.getAllParameterModels()).thenReturn(singletonList(parameterModel));
    when(parameterized.getParameterGroupModels()).thenReturn(singletonList(defaultGroup));
    componentBuilder.withParameterizedModel(parameterized);
  }

  private static class MockChainModel implements NestedChainModel {

    private final ChainExecutionOccurrence occurrence;

    public MockChainModel(ChainExecutionOccurrence occurrence) {
      this.occurrence = occurrence;
    }


    @Override
    public boolean isRequired() {
      return false;
    }

    @Override
    public Set<StereotypeModel> getAllowedStereotypes() {
      return null;
    }

    @Override
    public int getMinOccurs() {
      return 1;
    }

    @Override
    public Optional<Integer> getMaxOccurs() {
      return Optional.of(1);
    }

    @Override
    public void accept(NestableElementModelVisitor visitor) {

    }

    @Override
    public void accept(ComponentModelVisitor visitor) {

    }

    @Override
    public ComponentVisibility getVisibility() {
      return null;
    }

    @Override
    public List<? extends NestableElementModel> getNestedComponents() {
      return emptyList();
    }

    @Override
    public <T extends ModelProperty> Optional<T> getModelProperty(Class<T> propertyType) {
      return empty();
    }

    @Override
    public Set<ModelProperty> getModelProperties() {
      return emptySet();
    }

    @Override
    public Optional<DeprecationModel> getDeprecationModel() {
      return empty();
    }

    @Override
    public Optional<DisplayModel> getDisplayModel() {
      return empty();
    }

    @Override
    public Set<ErrorModel> getErrorModels() {
      return emptySet();
    }

    @Override
    public ChainExecutionOccurrence getChainExecutionOccurrence() {
      return occurrence;
    }

    @Override
    public List<ParameterGroupModel> getParameterGroupModels() {
      return emptyList();
    }

    @Override
    public String getDescription() {
      return null;
    }

    @Override
    public String getName() {
      return null;
    }

    @Override
    public StereotypeModel getStereotype() {
      return null;
    }

    @Override
    public Optional<MuleVersion> getMinMuleVersion() {
      return empty();
    }
  }

  private static class MockRouteModel implements NestedRouteModel, HasChainExecutionOccurrence {

    private final ChainExecutionOccurrence occurrence;

    public MockRouteModel(ChainExecutionOccurrence occurrence) {
      this.occurrence = occurrence;
    }

    @Override
    public ChainExecutionOccurrence getChainExecutionOccurrence() {
      return occurrence;
    }

    @Override
    public boolean isRequired() {
      return false;
    }

    @Override
    public int getMinOccurs() {
      return 1;
    }

    @Override
    public Optional<Integer> getMaxOccurs() {
      return Optional.of(1);
    }

    @Override
    public void accept(NestableElementModelVisitor visitor) {

    }

    @Override
    public void accept(ComponentModelVisitor visitor) {

    }

    @Override
    public ComponentVisibility getVisibility() {
      return null;
    }

    @Override
    public List<? extends NestableElementModel> getNestedComponents() {
      return emptyList();
    }

    @Override
    public <T extends ModelProperty> Optional<T> getModelProperty(Class<T> propertyType) {
      return empty();
    }

    @Override
    public Set<ModelProperty> getModelProperties() {
      return emptySet();
    }

    @Override
    public Optional<DeprecationModel> getDeprecationModel() {
      return empty();
    }

    @Override
    public Optional<DisplayModel> getDisplayModel() {
      return empty();
    }

    @Override
    public Set<ErrorModel> getErrorModels() {
      return emptySet();
    }

    @Override
    public List<ParameterGroupModel> getParameterGroupModels() {
      return emptyList();
    }

    @Override
    public String getDescription() {
      return null;
    }

    @Override
    public String getName() {
      return null;
    }

    @Override
    public StereotypeModel getStereotype() {
      return null;
    }

    @Override
    public Optional<MuleVersion> getMinMuleVersion() {
      return empty();
    }
  }

  private static class MockListOfRoutesModel implements NestedRouteModel, HasChainExecutionOccurrence {

    private final ChainExecutionOccurrence occurrence;

    public MockListOfRoutesModel(ChainExecutionOccurrence occurrence) {
      this.occurrence = occurrence;
    }

    @Override
    public ChainExecutionOccurrence getChainExecutionOccurrence() {
      return occurrence;
    }

    @Override
    public boolean isRequired() {
      return false;
    }

    @Override
    public int getMinOccurs() {
      return 1;
    }

    @Override
    public Optional<Integer> getMaxOccurs() {
      return Optional.empty();
    }

    @Override
    public void accept(NestableElementModelVisitor visitor) {

    }

    @Override
    public void accept(ComponentModelVisitor visitor) {

    }

    @Override
    public ComponentVisibility getVisibility() {
      return null;
    }

    @Override
    public List<? extends NestableElementModel> getNestedComponents() {
      return emptyList();
    }

    @Override
    public <T extends ModelProperty> Optional<T> getModelProperty(Class<T> propertyType) {
      return empty();
    }

    @Override
    public Set<ModelProperty> getModelProperties() {
      return emptySet();
    }

    @Override
    public Optional<DeprecationModel> getDeprecationModel() {
      return empty();
    }

    @Override
    public Optional<DisplayModel> getDisplayModel() {
      return empty();
    }

    @Override
    public Set<ErrorModel> getErrorModels() {
      return emptySet();
    }

    @Override
    public List<ParameterGroupModel> getParameterGroupModels() {
      return emptyList();
    }

    @Override
    public String getDescription() {
      return null;
    }

    @Override
    public String getName() {
      return null;
    }

    @Override
    public StereotypeModel getStereotype() {
      return null;
    }

    @Override
    public Optional<MuleVersion> getMinMuleVersion() {
      return empty();
    }
  }
}
