/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.cfg.utils;

import static java.util.Optional.of;

import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.error.ErrorTypeBuilder;
import org.mule.runtime.cfg.api.ChainExecutionPathTree;
import org.mule.runtime.cfg.api.ChainExecutionPathTreeFactory;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Stream;

import org.mockito.stubbing.Answer;

public class TestCfg {

  private final ChainExecutionPathTreeFactory cfgFactory;
  private final Map<String, ComponentAst> flows;

  public static TestCfgBuilder builder() {
    return new TestCfgBuilder();
  }

  public static class TestCfgBuilder {

    private final Map<String, ComponentAst> flows = new HashMap<>();

    public TestCfgBuilder addFlow(String flowName, Consumer<MockChainBuilder> flowConfigurer) {
      MockChainBuilder flowBuilder = MockChainBuilder.newBuilder(flowName);
      flowConfigurer.accept(flowBuilder);
      flows.put(flowName, flowBuilder.build());
      return this;
    }

    public TestCfg build() {
      ArtifactAst application = getApplication();
      when(application.topLevelComponentsStream())
          .then((Answer<Stream<ComponentAst>>) invocationOnMock -> flows.values().stream());
      return new TestCfg(new ChainExecutionPathTreeFactory(application), flows);
    }
  }

  private TestCfg(ChainExecutionPathTreeFactory cfgFactory, Map<String, ComponentAst> flows) {
    this.cfgFactory = cfgFactory;
    this.flows = flows;
  }

  public ChainExecutionPathTree getCfgForFlow(String flowName) {
    return cfgFactory.generateFor(flows.get(flowName));
  }

  private static ArtifactAst getApplication() {
    ArtifactAst application = mock(ArtifactAst.class);
    ErrorTypeRepository repository = mock(ErrorTypeRepository.class);

    ErrorType parent = ErrorTypeBuilder.builder().identifier("ANY").namespace("MULE").build();

    when(repository.lookupErrorType(any())).thenAnswer((Answer<Optional<ErrorType>>) invocationOnMock -> {
      ComponentIdentifier identifier = invocationOnMock.getArgument(0);
      ErrorTypeBuilder builder = ErrorTypeBuilder.builder().identifier(identifier.getName()).namespace(identifier.getNamespace());
      if (!identifier.getName().equals("ANY") || !identifier.getNamespace().equals("MULE")) {
        builder.parentErrorType(parent);
      }
      return of(builder.build());
    });
    when(application.getErrorTypeRepository()).thenReturn(repository);
    return application;
  }
}
