/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.connectivity.internal.persistence;

import org.mule.runtime.connectivity.api.platform.schema.ConnectivitySchemaParameter;
import org.mule.runtime.connectivity.internal.platform.schema.DefaultConnectivitySchemaParameter;

import java.io.IOException;
import java.util.List;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.TypeAdapter;
import com.google.gson.internal.Streams;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonWriter;

/**
 * {@link TypeAdapter} for instances of {@link ConnectivitySchemaParameter}.
 * <p>
 * Main purpose of this class is handling the {@link ConnectivitySchemaParameter#getEnumValues()} property which needs to be
 * renamed as {@code enum} at the json level.
 *
 * @since 1.1
 */
public class ConnectivitySchemaParameterTypeAdapter extends TypeAdapter<ConnectivitySchemaParameter> {

  private static final String ENUM_PROPERTY_NAME = "enumValues";
  private final Gson delegate;

  public ConnectivitySchemaParameterTypeAdapter() {
    this.delegate = new GsonBuilder()
        .registerTypeAdapterFactory(new DefaultImplementationTypeAdapterFactory<>(ConnectivitySchemaParameter.class,
                                                                                  DefaultConnectivitySchemaParameter.class))
        .create();
  }

  @Override
  public void write(JsonWriter out, ConnectivitySchemaParameter value) throws IOException {
    JsonObject object = (JsonObject) delegate.toJsonTree(value);
    List<String> enumValues = value.getEnumValues();
    if (enumValues == null || enumValues.isEmpty()) {
      object.remove(ENUM_PROPERTY_NAME);
    }

    Streams.write(object, out);
  }

  @Override
  public ConnectivitySchemaParameter read(JsonReader in) throws IOException {
    JsonObject object = (JsonObject) Streams.parse(in);
    JsonElement enumField = object.remove(ENUM_PROPERTY_NAME);
    if (enumField != null) {
      object.add(ENUM_PROPERTY_NAME, enumField);
    }
    return delegate.fromJson(object, ConnectivitySchemaParameter.class);
  }
}
