/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.connectivity.api.platform.schema.builder;

import org.mule.api.annotation.NoImplement;
import org.mule.runtime.connectivity.api.platform.schema.ConnectivitySchema;
import org.mule.runtime.connectivity.api.platform.schema.ExchangeAssetDescriptor;
import org.mule.runtime.connectivity.internal.platform.schema.builder.DefaultConnectivitySchemaBuilder;

import java.util.function.Consumer;

/**
 * Builder for creating instnaces of {@link ConnectivitySchema}
 *
 * @since 1.0
 */
@NoImplement
public interface ConnectivitySchemaBuilder {

  /**
   * @return a new builder instance
   */
  static ConnectivitySchemaBuilder newInstance() {
    return new DefaultConnectivitySchemaBuilder();
  }

  /**
   * Sets the schema's GAV
   *
   * @param groupId    a valid Maven {@code groupId}
   * @param artifactId a valid Maven {@code artifactId}
   * @param version    a valid Maven {@code version}
   * @return {@code this} builder
   */
  ConnectivitySchemaBuilder gav(String groupId, String artifactId, String version);

  /**
   * Specifies an identifier for the authentication type modeled by this schema.
   * <p>
   * This identifier should ideally be defined in the connectivity AML vocabulary.
   *
   * @param authenticationType the authentication type id
   * @return {@code this} builder
   */
  ConnectivitySchemaBuilder authenticationType(String authenticationType);

  /**
   * Specifies the name of the system targeted by this connection
   *
   * @param system the system name
   * @return {@code this} builder
   */
  ConnectivitySchemaBuilder system(String system);

  /**
   * Specifies the name of a Mule connector's {@code ConnectionProvider} which implements this schema
   *
   * @param connectionProviderName a ConnectionProvider name
   * @return {@code this} builder
   */
  ConnectivitySchemaBuilder connectionProviderName(String connectionProviderName);

  /**
   * Adds a label
   *
   * @param key   the label key
   * @param value the label value
   * @return {@code this} builder
   */
  ConnectivitySchemaBuilder withLabel(String key, String value);

  /**
   * Adds a dependant assert declaration
   *
   * @param assetDescriptor the asset descriptor
   * @return {@code this} builder
   */
  ConnectivitySchemaBuilder addAsset(ExchangeAssetDescriptor assetDescriptor);

  /**
   * Specifies a dependant vocabulary referenced in the produced schema
   *
   * @param ns  the vocabulary namespace
   * @param url the vocabulary url
   * @return {@code this} builder
   */
  ConnectivitySchemaBuilder uses(String ns, String url);

  /**
   * Specifies an external dependant vocabulary referenced in the produced schema
   *
   * @param ns  the vocabulary namespace
   * @param url the vocabulary url
   * @return {@code this} builder
   */
  ConnectivitySchemaBuilder usesExternal(String ns, String url);

  /**
   * Specifies the connection class term as defined in the Connectivity AML Vocabulary
   *
   * @param classTerm the connectivity class term
   * @return {@code this} builder
   */
  ConnectivitySchemaBuilder connectionClassTerm(String classTerm);

  /**
   * Adds a parameter into the range through the use of a {@link ConnectionParameterConfigurer}
   *
   * @param name               the parameter name
   * @param configurerConsumer a {@link Consumer} for a {@link ConnectionParameterConfigurer} on which the parameter is described
   * @return {@code this} builder
   */
  ConnectivitySchemaBuilder withParameter(String name, Consumer<ConnectionParameterConfigurer> configurerConsumer);

  /**
   * Specifies a schema specific custom range through the use of a {@link CustomRangeConfigurer}
   *
   * @param name               the range name
   * @param configurerConsumer a {@link Consumer} for a {@link CustomRangeConfigurer} on which the range is described
   * @return {@code this} builder
   */
  ConnectivitySchemaBuilder withCustomRange(String name, Consumer<CustomRangeConfigurer> configurerConsumer);

  /**
   * @return The produced in
   */
  ConnectivitySchema build();

  /**
   * A configurer for describing a custom Range
   *
   * @since 1.0
   */
  @NoImplement
  interface CustomRangeConfigurer {

    /**
     * Specifies the custom range class term. Ideally, this term is defined in the Connectivity AML Vocabulary.
     *
     * @param classTerm the class term.
     * @return {@code this} configurer
     */
    CustomRangeConfigurer setClassTerm(String classTerm);

    /**
     * Adds a parameter into the range through the use of a {@link ConnectionParameterConfigurer}
     *
     * @param name               the parameter name
     * @param configurerConsumer a {@link Consumer} for a {@link ConnectionParameterConfigurer} on which the parameter is
     *                           described
     * @return {@code this} configurer
     */
    CustomRangeConfigurer addParameter(String name, Consumer<ConnectionParameterConfigurer> configurerConsumer);
  }

  /**
   * A configurer for describing a parameter
   *
   * @since 4.4.0
   */
  @NoImplement
  interface ConnectionParameterConfigurer {

    /**
     * Specifies the parameter's property term. Ideally, this term is defined in the Connectivity AML Vocabulary.
     *
     * @param propertyTerm the property term.
     * @return {@code this} configurer
     */
    ConnectionParameterConfigurer setPropertyTerm(String propertyTerm);

    /**
     * Specifies the parameter's range
     *
     * @param range a range
     * @return {@code this} configurer
     */
    ConnectionParameterConfigurer setRange(String range);

    /**
     * Specifies whether the parameter is mandatory or not
     *
     * @param mandatory the mandatory boolean value
     * @return {@code this} configurer
     */
    ConnectionParameterConfigurer setMandatory(boolean mandatory);

    /**
     * Indicates that the parameter represents an enum by providing its supported {@code values}
     *
     * @param values the supported values.
     * @return {@code this} configurer
     * @since 1.1
     */
    ConnectionParameterConfigurer enumOf(String... values);
  }
}
