/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.connectivity.internal.platform.schema.builder;

import static java.util.Arrays.asList;

import org.mule.runtime.connectivity.api.platform.schema.ConnectivitySchema;
import org.mule.runtime.connectivity.api.platform.schema.ExchangeAssetDescriptor;
import org.mule.runtime.connectivity.api.platform.schema.builder.ConnectivitySchemaBuilder;
import org.mule.runtime.connectivity.internal.platform.schema.DefaultConnectivitySchema;
import org.mule.runtime.connectivity.internal.platform.schema.DefaultConnectivitySchemaNode;
import org.mule.runtime.connectivity.internal.platform.schema.DefaultConnectivitySchemaParameter;
import org.mule.runtime.connectivity.internal.platform.schema.DefaultConnectivitySchemaRoot;

import java.util.function.Consumer;

public class DefaultConnectivitySchemaBuilder implements ConnectivitySchemaBuilder {

  private final DefaultConnectivitySchema product = new DefaultConnectivitySchema();
  private final DefaultConnectivitySchemaNode connectionNode = new DefaultConnectivitySchemaNode();

  public DefaultConnectivitySchemaBuilder() {
    product.getDefinition().getNodeMappings().put("Connection", connectionNode);
    DefaultConnectivitySchemaRoot root = (DefaultConnectivitySchemaRoot) product.getDefinition().getDocument().getRoot();
    root.setEncodes("Connection");
    uses("connectivity", "anypoint://vocabulary/connectivity.yaml")
        .uses("core", "anypoint://vocabulary/core.yaml")
        .uses("apiContract", "anypoint://vocabulary/api-contract.yaml");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConnectivitySchemaBuilder gav(String groupId, String artifactId, String version) {
    product.setGroupId(groupId);
    product.setArtifactId(artifactId);
    product.setVersion(version);

    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConnectivitySchemaBuilder authenticationType(String authenticationType) {
    product.getLabels().put("type", authenticationType);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConnectivitySchemaBuilder system(String system) {
    product.getLabels().put("system", system);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConnectivitySchemaBuilder connectionProviderName(String connectionProviderName) {
    product.getLabels().put("connectionProvider", connectionProviderName);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConnectivitySchemaBuilder withLabel(String key, String value) {
    product.getLabels().put(key, value);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConnectivitySchemaBuilder addAsset(ExchangeAssetDescriptor assetDescriptor) {
    product.getAssets().add(assetDescriptor);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConnectivitySchemaBuilder uses(String ns, String url) {
    product.getDefinition().getUses().put(ns, url);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConnectivitySchemaBuilder usesExternal(String ns, String url) {
    product.getDefinition().getExternal().put(ns, url);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConnectivitySchemaBuilder connectionClassTerm(String classTerm) {
    connectionNode.setClassTerm(classTerm);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConnectivitySchemaBuilder withParameter(String name, Consumer<ConnectionParameterConfigurer> configurerConsumer) {
    DefaultConnectionParameterConfigurer configurer = new DefaultConnectionParameterConfigurer();
    configurerConsumer.accept(configurer);
    connectionNode.getMappings().put(name, configurer.product);

    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConnectivitySchemaBuilder withCustomRange(String name, Consumer<CustomRangeConfigurer> configurerConsumer) {
    DefaultCustomRangeConfigurer configurer = new DefaultCustomRangeConfigurer();
    configurerConsumer.accept(configurer);
    product.getDefinition().getNodeMappings().put(name, configurer.product);

    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConnectivitySchema build() {
    return product;
  }

  /**
   * A configurer for describing a custom Range
   *
   * @since 1.0
   */
  private static class DefaultCustomRangeConfigurer implements CustomRangeConfigurer {

    private DefaultConnectivitySchemaNode product = new DefaultConnectivitySchemaNode();

    /**
     * {@inheritDoc}
     */
    @Override
    public CustomRangeConfigurer setClassTerm(String classTerm) {
      product.setClassTerm(classTerm);
      return this;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public CustomRangeConfigurer addParameter(String name,
                                              Consumer<ConnectivitySchemaBuilder.ConnectionParameterConfigurer> configurerConsumer) {
      DefaultConnectionParameterConfigurer configurer = new DefaultConnectionParameterConfigurer();
      configurerConsumer.accept(configurer);
      product.getMappings().put(name, configurer.product);

      return this;
    }
  }

  private static class DefaultConnectionParameterConfigurer implements ConnectionParameterConfigurer {

    private DefaultConnectivitySchemaParameter product = new DefaultConnectivitySchemaParameter();

    /**
     * {@inheritDoc}
     */
    @Override
    public ConnectionParameterConfigurer setPropertyTerm(String propertyTerm) {
      product.setPropertyTerm(propertyTerm);
      return this;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public ConnectionParameterConfigurer setRange(String range) {
      product.setRange(range);
      return this;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public ConnectionParameterConfigurer setMandatory(boolean mandatory) {
      product.setMandatory(mandatory);
      return this;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public ConnectionParameterConfigurer enumOf(String... values) {
      if (values != null) {
        product.setEnumValues(asList(values));
      }

      return this;
    }
  }
}
