/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.connectivity.internal.platform.schema.generator;

import static java.util.Arrays.stream;
import static org.apache.commons.lang3.StringUtils.capitalize;
import static org.mule.metadata.api.utils.MetadataTypeUtils.getTypeId;
import static org.mule.runtime.api.util.NameUtils.hyphenize;
import static org.mule.runtime.connectivity.api.platform.schema.ConnectivityVocabulary.CONNECTION;
import static org.mule.runtime.connectivity.api.platform.schema.ConnectivityVocabulary.CONNECTIVITY_PREFIX;
import static org.mule.runtime.connectivity.internal.platform.schema.SemanticTermsHelper.getConnectionTerms;
import static org.mule.runtime.connectivity.internal.platform.schema.SemanticTermsHelper.getParameterTerms;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.metadata.api.annotation.EnumAnnotation;
import org.mule.metadata.api.model.BooleanType;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.NumberType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.StringType;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.connection.ConnectionProviderModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.util.IdempotentExtensionWalker;
import org.mule.runtime.api.util.Reference;
import org.mule.runtime.connectivity.api.platform.schema.ConnectivitySchema;
import org.mule.runtime.connectivity.api.platform.schema.ExchangeAssetDescriptor;
import org.mule.runtime.connectivity.api.platform.schema.builder.ConnectivitySchemaBuilder;
import org.mule.runtime.connectivity.api.platform.schema.generator.ConnectivitySchemaGenerator;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;

import org.slf4j.Logger;

/**
 * Default implementation of {@link ConnectivitySchemaGenerator}
 *
 * @since 1.0
 */
public class DefaultConnectivitySchemaGenerator implements ConnectivitySchemaGenerator {

  private static final Logger LOGGER = getLogger(DefaultConnectivitySchemaGenerator.class);
  public static final String SCHEMA_GROUP_ID = "com.mulesoft.schemas";

  private final Function<ConnectionProviderModel, Set<String>> connectionTermsExtractor;
  private final Function<ParameterModel, Set<String>> parameterTermsExtractor;
  private final Function<MetadataType, Set<String>> typeTermsExtractor;
  private final Function<ConnectionProviderModel, Boolean> connectionPredicate;
  private final BiFunction<ConnectionProviderModel, ParameterModel, Boolean> parameterPredicate;
  private final Map<MetadataType, String> customRanges = new HashMap<>();

  public DefaultConnectivitySchemaGenerator(Function<ConnectionProviderModel, Set<String>> connectionTermsExtractor,
                                            Function<ParameterModel, Set<String>> parameterTermsExtractor,
                                            Function<MetadataType, Set<String>> typeTermsExtractor,
                                            Function<ConnectionProviderModel, Boolean> connectionPredicate,
                                            BiFunction<ConnectionProviderModel, ParameterModel, Boolean> parameterPredicate) {
    this.connectionTermsExtractor = connectionTermsExtractor;
    this.parameterTermsExtractor = parameterTermsExtractor;
    this.typeTermsExtractor = typeTermsExtractor;
    this.connectionPredicate = connectionPredicate;
    this.parameterPredicate = parameterPredicate;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<ConnectivitySchema> generateSchemas(ExtensionModel extensionModel, ExchangeAssetDescriptor assetDescriptor) {
    List<ConnectivitySchema> schemas = new ArrayList<>();

    new IdempotentExtensionWalker() {

      @Override
      protected void onConnectionProvider(ConnectionProviderModel model) {
        if (connectionPredicate.apply(model)) {
          schemas.add(createSchema(extensionModel, model, assetDescriptor));
        }
      }
    }.walk(extensionModel);

    return schemas;
  }

  private ConnectivitySchema createSchema(ExtensionModel extensionModel,
                                          ConnectionProviderModel connectionProviderModel,
                                          ExchangeAssetDescriptor assetDescriptor) {
    final String connectionTerm = getConnectionTerm(connectionProviderModel);
    final String connectionType = extractConnectionType(connectionTerm);
    final ConnectivitySchemaBuilder builder = ConnectivitySchemaBuilder.newInstance();

    final String artifactId = buildSchemaArtifactId(assetDescriptor, connectionType);
    final String version = extensionModel.getVersion();

    LOGGER.debug("Declaring schema {}:{}:{}", SCHEMA_GROUP_ID, artifactId, version);

    builder.gav(SCHEMA_GROUP_ID, artifactId, version)
        .authenticationType(connectionType)
        .connectionClassTerm(connectionTerm)
        .connectionProviderName(connectionProviderModel.getName())
        .system(getSystemName(extensionModel))
        .addAsset(assetDescriptor);

    parseParameters(connectionProviderModel, builder);

    customRanges.clear();
    return builder.build();
  }

  private String extractConnectionType(String connectionTerm) {
    if (connectionTerm.startsWith(CONNECTIVITY_PREFIX)) {
      connectionTerm = connectionTerm.substring(CONNECTIVITY_PREFIX.length());
    }

    return connectionTerm;
  }

  private void parseParameters(ConnectionProviderModel model, ConnectivitySchemaBuilder builder) {
    Set<String> seenTerms = new HashSet<>();
    model.getAllParameterModels().forEach(param -> {
      if (!shouldIncludeInSchema(model, param)) {
        return;
      }
      builder.withParameter(param.getName(), attr -> {
        attr.setMandatory(param.isRequired())
            .setPropertyTerm(getParameterTerm(param, seenTerms))
            .setRange(asRange(param.getType(), builder, param.getName()));


        param.getType().getAnnotation(EnumAnnotation.class)
            .ifPresent(enumAnnotation -> attr.enumOf(stream(enumAnnotation.getValues()).toArray(String[]::new)));
      });
    });
  }

  private String buildSchemaArtifactId(ExchangeAssetDescriptor assetDescriptor, String connectionType) {
    return assetDescriptor.getAssetId() + ("-" + hyphenize(connectionType)).replaceAll("-o-auth-", "-oauth-");
  }

  private String asRange(MetadataType type, ConnectivitySchemaBuilder builder, String memberName) {
    Reference<String> term = new Reference<>();
    type.accept(new MetadataTypeVisitor() {

      @Override
      public void visitString(StringType stringType) {
        term.set("string");
      }

      @Override
      public void visitNumber(NumberType numberType) {
        term.set("number");
      }

      @Override
      public void visitBoolean(BooleanType booleanType) {
        term.set("boolean");
      }

      @Override
      public void visitObject(ObjectType objectType) {
        term.set(customRanges.computeIfAbsent(objectType, k -> {
          final String rangeName = getRangeName(objectType);
          declareCustomRange(objectType, rangeName, builder);
          return rangeName;
        }));
      }

      private String getRangeName(ObjectType objectType) {
        return getTypeId(objectType)
            .map(typeId -> {
              String[] classSegments = typeId.split("\\.");
              if (classSegments.length > 1) {
                typeId = classSegments[classSegments.length - 1];
              }

              return typeId;
            })
            .orElse(capitalize(memberName));
      }
    });

    return term.get();
  }

  private void declareCustomRange(ObjectType objectType, String rangeName, ConnectivitySchemaBuilder builder) {
    LOGGER.debug("Declaring custom range '{}'", rangeName);

    builder.withCustomRange(rangeName, range -> {
      range.setClassTerm(getTypeTerm(objectType, null, rangeName));
      objectType.getFields().forEach(field -> {
        MetadataType fieldType = field.getValue();

        if (!isSupported(fieldType)) {
          return;
        }

        String fieldName = field.getKey().getName().toString();
        Set<String> seenTerms = new HashSet<>();
        range.addParameter(fieldName,
                           attr -> attr.setPropertyTerm(getTypeTerm(field, seenTerms, fieldName))
                               .setMandatory(field.isRequired())
                               .setRange(asRange(fieldType, builder, fieldName)));
      });
    });
  }

  private boolean shouldIncludeInSchema(ConnectionProviderModel connectionProviderModel, ParameterModel parameterModel) {
    if (!parameterPredicate.apply(connectionProviderModel, parameterModel)) {
      return false;
    }

    if (parameterModel.getName().toLowerCase().contains("timeout")) {
      return false;
    }

    return isSupported(parameterModel.getType())
        ? !isInfrastructure(parameterModel)
        : false;
  }

  private boolean isInfrastructure(ParameterModel parameterModel) {
    return parameterModel.getModelProperties().stream().anyMatch(p -> "infrastructureParameter".equals(p.getName()));
  }

  private boolean isSupported(MetadataType type) {
    if (type instanceof StringType
        || type instanceof NumberType
        || type instanceof BooleanType) {
      return true;
    }

    if (type instanceof ObjectType) {
      return !((ObjectType) type).getFields().isEmpty();
    }

    return false;
  }

  private String getConnectionTerm(ConnectionProviderModel model) {
    return getConnectionTerms(connectionTermsExtractor.apply(model)).stream()
        .filter(t -> t.endsWith("Connection"))
        .findFirst()
        .orElse(CONNECTION);
  }

  private String getParameterTerm(ParameterModel model, Set<String> seenTerms) {
    return getTermForProperty(getParameterTerms(parameterTermsExtractor.apply(model)), seenTerms, model.getName());
  }

  private String getTermForProperty(Set<String> terms, Set<String> seenTerms, String propertyName) {
    if (!terms.isEmpty()) {
      for (String term : terms) {
        if (seenTerms == null) {
          return term;
        } else if (seenTerms.add(term)) {
          return term;
        } else {
          LOGGER
              .debug("Property {} contains term {} which has already been used in its owner node. Term discarded for this property",
                     propertyName);
        }
      }
    }

    LOGGER.debug("No property term could be determined for Parameter {}", propertyName);
    return null;
  }

  private String getTypeTerm(MetadataType type, Set<String> seenTerms, String typeName) {
    return getTermForProperty(getParameterTerms(typeTermsExtractor.apply(type)), seenTerms, typeName);
  }

  private String getSystemName(ExtensionModel extensionModel) {
    return extensionModel.getName()
        .replaceAll("Connector", "")
        .replaceAll("connector", "")
        .replaceAll("Mule", "")
        .replaceAll("mule", "")
        .trim();
  }

}
