/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.core.api.util;

import static org.mule.runtime.core.api.util.ObjectUtils.getBoolean;
import static org.mule.runtime.core.api.util.ObjectUtils.getByte;
import static org.mule.runtime.core.api.util.ObjectUtils.getDouble;
import static org.mule.runtime.core.api.util.ObjectUtils.getFloat;
import static org.mule.runtime.core.api.util.ObjectUtils.getInt;
import static org.mule.runtime.core.api.util.ObjectUtils.getLong;
import static org.mule.runtime.core.api.util.ObjectUtils.getShort;
import static org.mule.runtime.core.api.util.ObjectUtils.getString;
import static org.mule.runtime.core.api.util.ObjectUtils.identityToShortString;

import static org.apache.commons.lang3.JavaVersion.JAVA_20;
import static org.apache.commons.lang3.StringUtils.split;
import static org.apache.commons.lang3.SystemUtils.isJavaVersionAtMost;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assume.assumeTrue;

import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.size.SmallTest;

import java.util.HashMap;

import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;

@SmallTest
public class ObjectUtilsTestCase extends AbstractMuleTestCase {

  @BeforeAll
  public static void checkJvmVersion() {
    assumeTrue(isJavaVersionAtMost(JAVA_20));
  }

  @Test
  void testIdentityToShortString() {
    assertEquals("null", identityToShortString(null));

    String source = "foo";
    String description = identityToShortString(source);
    String[] components = split(description, '@');

    assertNotNull(components);
    assertEquals(2, components.length);
    assertEquals("String", components[0]);
    assertEquals(Integer.toHexString(System.identityHashCode(source)), components[1]);
  }

  @Test
  void testBooleanConversion() throws Exception {
    Object value = "true";
    assertTrue(getBoolean(value, false));

    value = "xyz";
    assertFalse(getBoolean(value, false));

    value = new Integer(6);
    assertTrue(getBoolean(value, false));

    value = new Integer(1);
    assertTrue(getBoolean(value, false));

    value = new Integer(0);
    assertFalse(getBoolean(value, false));

    value = new Integer(-41);
    assertFalse(getBoolean(value, false));

    value = null;
    assertFalse(getBoolean(value, false));
  }

  @Test
  void testShortConversion() throws Exception {
    Object value = "123";
    assertEquals((short) 123, getShort(value, (short) -1));

    value = "xyz";
    assertEquals((short) -1, getShort(value, (short) -1));

    value = new Integer(6);
    assertEquals((short) 6, getShort(value, (short) -1));

    value = new Double(63.4);
    assertEquals((short) 63, getShort(value, (short) -1));

    value = new Float(-163.2);
    assertEquals((short) -163, getShort(value, (short) -1));

    value = null;
    assertEquals((short) -1, getShort(value, (short) -1));

  }

  @Test
  void testByteConversion() throws Exception {
    Object value = "123";
    assertEquals((byte) 123, getByte(value, (byte) -1));

    value = "xyz";
    assertEquals((byte) -1, getByte(value, (byte) -1));

    value = new Integer(6);
    assertEquals((byte) 6, getByte(value, (byte) -1));

    value = new Double(63.4);
    assertEquals((byte) 63, getByte(value, (byte) -1));

    value = new Float(-163.2);
    assertEquals((byte) -163, getByte(value, (byte) -1));

    value = null;
    assertEquals((byte) -1, getByte(value, (byte) -1));
  }

  @Test
  void testIntConversion() throws Exception {
    Object value = "123";
    assertEquals(123, getInt(value, -1));

    value = "xyz";
    assertEquals(-1, getInt(value, -1));

    value = new Integer(6);
    assertEquals(6, getInt(value, -1));

    value = new Double(63.4);
    assertEquals(63, getInt(value, -1));

    value = new Float(-163.2);
    assertEquals(-163, getInt(value, -1));

    value = null;
    assertEquals(-1, getInt(value, -1));
  }

  @Test
  void testLongConversion() throws Exception {
    Object value = "123";
    assertEquals(123, getLong(value, -1l));

    value = "xyz";
    assertEquals(-1, getLong(value, -1l));

    value = new Integer(6);
    assertEquals(6, getLong(value, -1l));

    value = new Double(63.4);
    assertEquals(63, getLong(value, -1l));

    value = new Float(-163.2);
    assertEquals(-163, getLong(value, -1l));

    value = null;
    assertEquals(-1, getLong(value, -1l));
  }

  @Test
  void testFloatConversion() throws Exception {
    Object value = "123.34";
    assertEquals(123.34, getFloat(value, -1f), 0.1f);

    value = "xyz";
    assertEquals(-1, getFloat(value, -1f), 0.1f);

    value = new Integer(6);
    assertEquals(6, getFloat(value, -1f), 0.1f);

    value = new Double(63.4);
    assertEquals(63.4, getFloat(value, -1f), 0.1f);

    value = new Float(-163.2);
    assertEquals(-163.2, getFloat(value, -1f), 0.1f);

    value = null;
    assertEquals(-1, getFloat(value, -1f), 0.1f);
  }

  @Test
  void testDoubleConversion() throws Exception {
    Object value = "123.34";
    assertEquals(123.34, getDouble(value, -1.0), 0.1d);

    value = "xyz";
    assertEquals(-1, getDouble(value, -1.0), 0.1d);

    value = new Integer(6);
    assertEquals(6, getDouble(value, -1.0), 0.1d);

    value = new Double(63.4);
    assertEquals(63.4, getDouble(value, -1.0), 0.1d);

    value = new Float(-163.2);
    assertEquals(-163.2, getDouble(value, -1.0), 0.1d);

    value = null;
    assertEquals(-1, getDouble(value, -1.0), 0.1d);
  }

  @Test
  void testStringConversion() throws Exception {
    Object value = "hello";
    assertEquals("hello", getString(value, "x"));

    value = new HashMap();
    assertEquals(new HashMap().toString(), getString(value, "x"));

    value = null;
    assertEquals("x", getString(value, "x"));
  }
}
