/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.embedded.internal;

import static com.google.common.base.Preconditions.checkState;
import static java.lang.Thread.currentThread;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.concurrent.Executors.newSingleThreadExecutor;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.io.FileUtils.copyInputStreamToFile;
import static org.codehaus.plexus.util.FileUtils.fileWrite;
import static org.codehaus.plexus.util.FileUtils.toFile;
import static org.mule.maven.client.api.MavenClientProvider.discoverProvider;
import static org.mule.maven.client.api.model.BundleScope.PROVIDED;
import static org.mule.runtime.module.embedded.internal.Serializer.serialize;
import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.MavenClientProvider;
import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.runtime.module.embedded.api.ArtifactConfiguration;
import org.mule.runtime.module.embedded.api.ContainerConfiguration;
import org.mule.runtime.module.embedded.api.ContainerInfo;
import org.mule.runtime.module.embedded.api.DeploymentService;
import org.mule.runtime.module.embedded.api.EmbeddedContainer;
import org.mule.runtime.module.embedded.api.Product;
import org.mule.runtime.module.embedded.internal.classloading.FilteringClassLoader;
import org.mule.runtime.module.embedded.internal.classloading.JdkOnlyClassLoaderFactory;

import com.google.gson.Gson;
import com.google.gson.JsonObject;

import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.stream.Collectors;

import org.codehaus.plexus.util.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultEmbeddedContainerBuilder implements EmbeddedContainer.EmbeddedContainerBuilder {

  private static Logger LOGGER = LoggerFactory.getLogger(DefaultEmbeddedContainerBuilder.class);

  private String muleVersion;
  private URI log4jConfigurationFile;
  private MavenConfiguration mavenConfiguration;
  private ContainerConfiguration containerConfiguration;
  private Product product;

  @Override
  public EmbeddedContainer.EmbeddedContainerBuilder muleVersion(String muleVersion) {
    this.muleVersion = muleVersion;
    return this;
  }

  @Override
  public EmbeddedContainer.EmbeddedContainerBuilder product(Product product) {
    this.product = product;
    return this;
  }

  @Override
  public EmbeddedContainer.EmbeddedContainerBuilder containerConfiguration(ContainerConfiguration containerConfiguration) {
    this.containerConfiguration = containerConfiguration;
    return this;
  }

  @Override
  public EmbeddedContainer.EmbeddedContainerBuilder log4jConfigurationFile(URI log4JConfigurationFile) {
    this.log4jConfigurationFile = log4JConfigurationFile;
    return this;
  }

  @Override
  public EmbeddedContainer.EmbeddedContainerBuilder mavenConfiguration(MavenConfiguration mavenConfiguration) {
    this.mavenConfiguration = mavenConfiguration;
    return this;
  }

  @Override
  public EmbeddedContainer build() {
    checkState(muleVersion != null, "muleVersion cannot be null");
    checkState(mavenConfiguration != null, "mavenConfiguration cannot be null");
    checkState(containerConfiguration != null, "containerConfiguration cannot be null");
    checkState(log4jConfigurationFile != null, "log4jConfigurationFile cannot be null");
    checkState(product != null, "product cannot be null");
    try {
      URL containerBaseFolder = containerConfiguration.getContainerFolder().toURI().toURL();

      if (log4jConfigurationFile != null) {
        File confDir = new File(FileUtils.toFile(containerBaseFolder), "conf");
        confDir.mkdirs();
        try (InputStream log4jConfFileInputStream = log4jConfigurationFile.toURL().openStream()) {
          copyInputStreamToFile(log4jConfFileInputStream, new File(confDir, "log4j2.xml"));
        }
      }

      FilteringClassLoader jdkOnlyClassLoader = JdkOnlyClassLoaderFactory.create();
      MavenClientProvider mavenClientProvider = discoverProvider(getClass().getClassLoader());
      MavenClient mavenClient = mavenClientProvider.createMavenClient(mavenConfiguration);

      persistMavenConfiguration(containerBaseFolder, mavenConfiguration);

      MavenContainerClassLoaderFactory classLoaderFactory = new MavenContainerClassLoaderFactory(mavenClient);

      try {
        return new EmbeddedContainer() {

          private boolean started = false;

          private Object embeddedController;
          private ClassLoader containerModulesClassLoader;
          private ClassLoader embeddedControllerBootstrapClassLoader;

          // Executor service to use for running the container operations. The agent is using jetty which uses thread locals that end up
          // with references to class loaders of the container and that causes a memory leak if the thread is never disposed. This
          // guarantees
          // that every operation happens in the thread of this executor service or a thread created by the runtime.
          private ExecutorService executorService;

          @Override
          public synchronized void start() {
            if (!started) {
              try {
                containerModulesClassLoader =
                    classLoaderFactory.create(muleVersion, product, jdkOnlyClassLoader, containerBaseFolder.toURI().toURL());

                embeddedControllerBootstrapClassLoader =
                    createEmbeddedImplClassLoader(containerModulesClassLoader, mavenClient, muleVersion, product);

                List<URL> services = classLoaderFactory.getServices(muleVersion, product);
                List<URL> serverPlugins = getServerPlugins(mavenClient, containerConfiguration.getServerPlugins());
                ContainerInfo containerInfo = new ContainerInfo(muleVersion, containerBaseFolder, services, serverPlugins);

                Class<?> controllerClass =
                    embeddedControllerBootstrapClassLoader.loadClass("org.mule.runtime.module.embedded.impl.EmbeddedController");
                Constructor<?> constructor = controllerClass.getConstructor(byte[].class);
                ByteArrayOutputStream containerOutputStream = new ByteArrayOutputStream(512);
                serialize(containerInfo, containerOutputStream);

                executorService = newSingleThreadExecutor(runnable -> new Thread(runnable, "Embedded"));

                embeddedController =
                    constructor.newInstance(containerOutputStream.toByteArray());

              } catch (Exception e) {
                // Clean up resources by calling stop in case of error during start
                doStop();
                throw new IllegalStateException("Cannot start embedded container", e);
              }
              started = true;
              executeUsingReflection(() -> embeddedController.getClass().getMethod("start"), emptyList());
            } else {
              if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Embedded container already started");
              }
            }
          }

          @Override
          public synchronized void stop() {
            if (started) {
              try {
                doStop();
              } finally {
                started = false;
              }
            } else {
              if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Embedded container already stopped");
              }
            }
          }

          private void doStop() {
            if (embeddedController != null) {
              executeUsingReflection(() -> embeddedController.getClass().getMethod("stop"), emptyList());
              embeddedController = null;
            }
            if (executorService != null) {
              executorService.shutdownNow();
              executorService = null;
            }

            closeClassLoader(containerModulesClassLoader);
            containerModulesClassLoader = null;
            closeClassLoader(embeddedControllerBootstrapClassLoader);
            embeddedControllerBootstrapClassLoader = null;
          }

          private void closeClassLoader(ClassLoader classLoader) {
            if (classLoader instanceof Closeable) {
              try {
                ((Closeable) classLoader).close();
              } catch (IOException e) {
                // Do nothing.
                if (LOGGER.isDebugEnabled()) {
                  LOGGER.debug("Failure closing container classloader", e);
                }
              }
            }
          }

          @Override
          public DeploymentService getDeploymentService() {
            return new DeploymentService() {

              @Override
              public void deployApplication(ArtifactConfiguration artifactConfiguration) {
                executeUsingReflection(() -> findEmbeddedMethod("deployApplication"),
                                       singletonList(artifactConfiguration));
              }

              @Override
              public void undeployApplication(String applicationName) {
                executeUsingReflection(() -> findEmbeddedMethod("undeployApplication"),
                                       singletonList(applicationName));
              }

              @Override
              public void deployDomain(ArtifactConfiguration artifactConfiguration) {
                executeUsingReflection(() -> findEmbeddedMethod("deployDomain"),
                                       singletonList(artifactConfiguration));
              }

              @Override
              public void undeployDomain(String domainName) {
                executeUsingReflection(() -> findEmbeddedMethod("undeployDomain"),
                                       singletonList(domainName));
              }
            };
          }

          @Override
          public File getContainerFolder() {
            return containerConfiguration.getContainerFolder();
          }

          private Method findEmbeddedMethod(String methodName) {
            Method[] methods = embeddedController.getClass().getMethods();
            for (Method method : methods) {
              if (method.getName().equals(methodName)) {
                return method;
              }
            }
            throw new RuntimeException(new NoSuchMethodException("Method " + methodName + " not found"));
          }

          private void executeUsingReflection(UncheckedSupplier<Method> methodSupplier, List<Serializable> args) {
            Future<?> future = executorService.submit(() -> {
              ClassLoader contextClassLoader = currentThread().getContextClassLoader();
              try {
                currentThread().setContextClassLoader(containerModulesClassLoader);

                Method method = methodSupplier.get();
                List<byte[]> arguments = args.stream().map(arg -> {
                  ByteArrayOutputStream byteOutputStream = new ByteArrayOutputStream();
                  serialize(arg, byteOutputStream);
                  return byteOutputStream.toByteArray();
                }).collect(toList());

                method.invoke(embeddedController, arguments.toArray(new Object[arguments.size()]));
              } catch (InvocationTargetException e) {
                Throwable cause = e.getCause();
                if (cause instanceof RuntimeException) {
                  throw (RuntimeException) cause;
                } else {
                  throw new IllegalStateException(cause);
                }
              } catch (Exception e) {
                throw new IllegalStateException(e);
              } finally {
                currentThread().setContextClassLoader(contextClassLoader);
              }
            });
            try {
              future.get();
            } catch (Exception e) {
              throw new RuntimeException(e);
            }
          }
        };
      } catch (Exception e) {
        throw new IllegalStateException("Cannot create embedded container", e);
      }
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
  }


  private List<URL> getServerPlugins(MavenClient mavenClient, List<BundleDescriptor> serverPlugins) {
    return serverPlugins.stream().map(serverPluginDescriptor -> {
      BundleDependency bundleDependency = mavenClient.resolveBundleDescriptor(serverPluginDescriptor);
      try {
        return bundleDependency.getBundleUri().toURL();
      } catch (MalformedURLException e) {
        throw new RuntimeException(e);
      }
    }).collect(Collectors.toList());
  }

  private void persistMavenConfiguration(URL containerBaseFolder, MavenConfiguration mavenConfiguration) throws IOException {
    File configurationFolder = new File(toFile(containerBaseFolder), "conf");
    if (!configurationFolder.exists()) {
      if (!configurationFolder.mkdirs()) {
        throw new IllegalArgumentException("Could not create MULE_HOME/conf folder in: " + configurationFolder.getAbsolutePath());
      }
    }

    JsonObject rootObject = new JsonObject();
    JsonObject muleRuntimeConfigObject = new JsonObject();
    rootObject.add("muleRuntimeConfig", muleRuntimeConfigObject);
    JsonObject mavenObject = new JsonObject();
    muleRuntimeConfigObject.add("maven", mavenObject);
    if (!mavenConfiguration.getMavenRemoteRepositories().isEmpty()) {
      JsonObject repositoriesObject = new JsonObject();
      mavenObject.add("repositories", repositoriesObject);
      mavenConfiguration.getMavenRemoteRepositories().forEach(mavenRepo -> {
        JsonObject repoObject = new JsonObject();
        repositoriesObject.add(mavenRepo.getId(), repoObject);
        repoObject.addProperty("url", mavenRepo.getUrl().toString());
        mavenRepo.getAuthentication().ifPresent(authentication -> {
          repoObject.addProperty("username", authentication.getUsername());
          repoObject.addProperty("password", authentication.getPassword());
        });
      });
    }
    mavenObject.addProperty("repositoryLocation", mavenConfiguration.getLocalMavenRepositoryLocation().getAbsolutePath());
    mavenConfiguration.getUserSettingsLocation().ifPresent(userSettingsLocation -> mavenObject
        .addProperty("userSettingsLocation", userSettingsLocation.getAbsolutePath()));
    mavenConfiguration.getGlobalSettingsLocation().ifPresent(globalSettingsLocation -> mavenObject
        .addProperty("globalSettingsLocation", globalSettingsLocation.getAbsolutePath()));
    mavenObject.addProperty("ignoreArtifactDescriptorRepositories", mavenConfiguration.getIgnoreArtifactDescriptorRepositories());
    mavenConfiguration.getSettingsSecurityLocation().ifPresent(securitySettingsLocation -> mavenObject
        .addProperty("settingsSecurityLocation", securitySettingsLocation.getAbsolutePath()));

    mavenObject.addProperty("forcePolicyUpdateNever", mavenConfiguration.getForcePolicyUpdateNever());
    mavenObject.addProperty("offLineMode", mavenConfiguration.getOfflineMode());

    String muleConfigContent = new Gson().toJson(rootObject);
    fileWrite(new File(configurationFolder, "mule-config.json"), muleConfigContent);
  }

  private static ClassLoader createEmbeddedImplClassLoader(ClassLoader parentClassLoader, MavenClient mavenClient,
                                                           String muleVersion, Product product)
      throws MalformedURLException {

    BundleDescriptor embeddedBomDescriptor = new BundleDescriptor.Builder().setGroupId("org.mule.distributions")
        .setArtifactId("mule-module-embedded-impl-bom").setVersion(muleVersion).setType("pom").build();

    BundleDescriptor embeddedImplDescriptor = new BundleDescriptor.Builder().setGroupId("org.mule.distributions")
        .setArtifactId("mule-module-embedded-impl").setVersion(muleVersion).setType("jar").build();

    BundleDependency embeddedBundleImplDescriptor = mavenClient.resolveBundleDescriptor(embeddedImplDescriptor);

    List<BundleDependency> embeddedImplDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, embeddedBomDescriptor);

    List<URL> embeddedUrls = embeddedImplDependencies.stream()
        .filter(bundleDependency -> !bundleDependency.getScope().equals(PROVIDED))
        .map(dep -> {
          try {
            return dep.getBundleUri().toURL();
          } catch (MalformedURLException e) {
            throw new RuntimeException(e);
          }
        })
        .collect(toList());
    embeddedUrls.add(embeddedBundleImplDescriptor.getBundleUri().toURL());

    return new URLClassLoader(embeddedUrls.toArray(new URL[embeddedUrls.size()]), parentClassLoader);
  }

  interface UncheckedSupplier<T> {

    T get() throws Exception;

  }

}
