/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.embedded.internal.jvm;

import static org.mule.runtime.module.embedded.internal.jvm.JvmVersionInterval.normalizeJava8OrLowerVersion;

import static java.util.regex.Pattern.compile;
import static java.util.stream.Collectors.joining;

import static com.vdurmont.semver4j.Semver.SemverType.LOOSE;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.vdurmont.semver4j.Semver;

/**
 * Represents a union of intervals of supported Java versions.
 * <p>
 * Implementation is immutable and thread-safe.
 *
 * @since 1.6
 */
public class JvmVersionIntervalUnion {

  private static final Pattern INTERVAL_PATTERN = compile("(\\[[^\\)]*\\))");

  /**
   * Parses a list of comma-separated intervals of supported Java versions. Interval endpoints may be open or closed, indicated by
   * parenthesis or brackets respectively.
   * <p>
   * Example: {@code [1.8.0_181,9),[11,12),[17,18)}
   *
   * @param intervalUnionString the string to parse.
   * @return the parsed JvmVersionIntervalUnion
   */
  public static JvmVersionIntervalUnion parse(String intervalUnionString) {
    final Matcher intervalUnionMatcher = INTERVAL_PATTERN.matcher(intervalUnionString);

    List<JvmVersionInterval> intervals = new ArrayList<>();
    while (intervalUnionMatcher.find()) {
      intervals.add(JvmVersionInterval.parse(intervalUnionMatcher.group()));
    }

    return new JvmVersionIntervalUnion(intervals);
  }

  private final List<JvmVersionInterval> intervals;

  /**
   * Creates an {@link JvmVersionIntervalUnion} containing the provided intervals.
   *
   * @param intervals the intervals to check against.
   */
  public JvmVersionIntervalUnion(List<JvmVersionInterval> intervals) {
    this.intervals = intervals;
  }

  /**
   * Checks if any of the intervals in this union includes the version indicated by {@code versionString}.
   *
   * @param versionString the version to check for inclusion in the intervals.
   * @return {@code true} if any interval contains the version indicated by {@code versionString}, {@code false} otherwise.
   */
  public boolean includes(String versionString) {
    return includes(new Semver(normalizeJava8OrLowerVersion(versionString), LOOSE).toStrict());
  }

  private boolean includes(final Semver version) {
    return intervals.stream().anyMatch(interval -> interval.includes(version));
  }

  @Override
  public String toString() {
    return intervals.stream().map(JvmVersionInterval::toString).collect(joining(",", "", ""));
  }
}
