/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.runtime.module.embedded.api;

import org.mule.api.annotation.NoImplement;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.runtime.module.embedded.internal.DefaultEmbeddedContainerBuilder;

import java.io.File;
import java.net.URI;

/**
 * Interface that represents an embedded container
 *
 * @since 4.0
 */
@NoImplement
public interface EmbeddedContainer {

  /**
   * Starts the container.
   */
  void start();

  /**
   * Stops the container.
   */
  void stop();

  /**
   * @return the version if the Mule runtime this embedded container is running.
   */
  String getMuleContainerVersion();

  /**
   * @return service to deploy and undeploy artifacts
   */
  DeploymentService getDeploymentService();

  /**
   * The location of the mule container base folder
   *
   * @return location of the container folder.
   */
  File getContainerFolder();

  /**
   * @return a new builder to create an {@link EmbeddedContainer}.
   */
  static EmbeddedContainerBuilder builder() {
    return new DefaultEmbeddedContainerBuilder();
  }


  /**
   * Builder for {@link EmbeddedContainer} instances. To create an instance of this builder use
   * {@link EmbeddedContainer#builder()} method
   */
  interface EmbeddedContainerBuilder {

    /**
     * @param muleVersion mule version to use for running the artifact.
     * @return same builder
     */
    EmbeddedContainerBuilder muleVersion(String muleVersion);

    /**
     * @param product the mule product to use
     * @return same builder
     */
    EmbeddedContainerBuilder product(Product product);

    /**
     * @param containerConfiguration configuration of the container
     * @return same builder
     */
    EmbeddedContainerBuilder containerConfiguration(ContainerConfiguration containerConfiguration);

    /**
     * Customizes the log4j configuration file for the artifact.
     *
     * @param log4JConfigurationFile absolute path to the log4j configuration file.
     * @return same builder
     */
    EmbeddedContainerBuilder log4jConfigurationFile(URI log4JConfigurationFile);

    /**
     * Customizes the maven configuration for the execution.
     *
     * @param mavenConfiguration maven configuration.
     * @return same buildeer
     */
    EmbeddedContainerBuilder mavenConfiguration(MavenConfiguration mavenConfiguration);

    /**
     * @return creates a new {@link EmbeddedContainer} with the provided configuration.
     */
    EmbeddedContainer build();

  }

}
