/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.embedded.api;

import org.mule.api.annotation.NoImplement;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.runtime.module.embedded.internal.DefaultEmbeddedContainerBuilder;

import java.io.File;
import java.net.URI;

/**
 * Interface that represents an embedded container
 *
 * @since 4.0
 */
@NoImplement
public interface EmbeddedContainer {

  /**
   * Starts the container.
   */
  void start();

  /**
   * Stops the container.
   */
  void stop();

  /**
   * @return the version if the Mule runtime this embedded container is running.
   */
  String getMuleContainerVersion();

  /**
   * @return service to deploy and undeploy artifacts
   */
  DeploymentService getDeploymentService();

  /**
   * The location of the mule container base folder
   *
   * @return location of the container folder.
   */
  File getContainerFolder();

  /**
   * Queries Java support metadata of the Mule runtime for the version specified by #getMuleContainerVersion() and checks it
   * against the currently running JVM.
   * <p>
   * Implementations are not required to do any kind of caching of the result.
   *
   * @return @{code true} if the version of the currently running JVM is supported.
   * @since 1.6
   */
  boolean isCurrentJvmVersionSupported();


  /**
   * Queries Java support metadata of the Mule runtime for the version specified by #getMuleContainerVersion() and checks it
   * against the currently running JVM.
   * <p>
   * Implementations are not required to do any kind of caching of the result.
   *
   * @return @{code true} if the version of the currently running JVM is recommended.
   * @since 1.6
   */
  boolean isCurrentJvmVersionRecommended();

  /**
   * @return a new builder to create an {@link EmbeddedContainer}.
   */
  static EmbeddedContainerBuilder builder() {
    return new DefaultEmbeddedContainerBuilder();
  }


  /**
   * Builder for {@link EmbeddedContainer} instances. To create an instance of this builder use
   * {@link EmbeddedContainer#builder()} method
   */
  @NoImplement
  interface EmbeddedContainerBuilder {

    /**
     * @param muleVersion mule version to use for running the artifact.
     * @return same builder
     */
    EmbeddedContainerBuilder muleVersion(String muleVersion);

    /**
     * @param product the mule product to use
     * @return same builder
     */
    EmbeddedContainerBuilder product(Product product);

    /**
     * @param containerConfiguration configuration of the container
     * @return same builder
     */
    EmbeddedContainerBuilder containerConfiguration(ContainerConfiguration containerConfiguration);

    /**
     * Customizes the log4j configuration file for the artifact.
     *
     * @param log4JConfigurationFile absolute path to the log4j configuration file.
     * @return same builder
     */
    EmbeddedContainerBuilder log4jConfigurationFile(URI log4JConfigurationFile);

    /**
     * Customizes the maven configuration for the execution.
     *
     * @param mavenConfiguration maven configuration.
     * @return same buildeer
     */
    EmbeddedContainerBuilder mavenConfiguration(MavenConfiguration mavenConfiguration);

    /**
     * Configures the container to run in a different class loader/layer than {@code mule-embedded-api} does.
     *
     * @param isolate whether to isolate the container.
     * @return same builder
     */
    EmbeddedContainerBuilder useIsolation(boolean isolate);

    /**
     * @return creates a new {@link EmbeddedContainer} with the provided configuration.
     */
    EmbeddedContainer build();

  }

}
