/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.embedded.internal;

import static org.mule.maven.pom.parser.api.model.BundleScope.PROVIDED;
import static org.mule.runtime.module.embedded.internal.jvm.JvmVersionIntervalUnion.parse;

import static java.lang.System.getProperty;
import static java.lang.Thread.currentThread;
import static java.util.concurrent.Executors.newSingleThreadExecutor;
import static java.util.stream.Collectors.toList;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.maven.client.api.MavenClient;
import org.mule.maven.pom.parser.api.model.BundleDependency;
import org.mule.maven.pom.parser.api.model.BundleDescriptor;
import org.mule.runtime.module.embedded.api.ContainerConfiguration;
import org.mule.runtime.module.embedded.api.ContainerInfo;
import org.mule.runtime.module.embedded.api.DeploymentService;
import org.mule.runtime.module.embedded.api.EmbeddedContainer;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;

import org.slf4j.Logger;

/**
 * Common functionality for implementations of {@link EmbeddedContainer}.
 *
 * @param <T> the type of embedded controller used by the implementation.
 */
public abstract class AbstractEmbeddedContainer<T> implements EmbeddedContainer {

  protected static final Logger LOGGER = getLogger(AbstractEmbeddedContainer.class);

  private final String muleVersion;
  private final ContainerConfiguration containerConfiguration;
  private final URL containerBaseFolder;
  private final MavenClient mavenClient;
  private final List<URL> services;

  private ClassLoader containerModulesClassLoader;
  private boolean started = false;
  private T embeddedController;
  private ClassLoader embeddedControllerBootstrapClassLoader;

  // Executor service to use for running the container operations. The agent is using jetty which uses thread locals that
  // end up
  // with references to class loaders of the container and that causes a memory leak if the thread is never disposed. This
  // guarantees
  // that every operation happens in the thread of this executor service or a thread created by the runtime.
  private ExecutorService executorService;

  public AbstractEmbeddedContainer(String muleVersion, ContainerConfiguration containerConfiguration,
                                   ClassLoader containerModulesClassLoader,
                                   List<URL> services, URL containerBaseFolder, MavenClient mavenClient) {
    this.muleVersion = muleVersion;
    this.containerConfiguration = containerConfiguration;
    this.containerModulesClassLoader = containerModulesClassLoader;
    this.services = services;
    this.containerBaseFolder = containerBaseFolder;
    this.mavenClient = mavenClient;
  }

  @Override
  public synchronized void start() {
    if (!started) {
      try {
        embeddedControllerBootstrapClassLoader =
            createEmbeddedImplClassLoader(containerModulesClassLoader, mavenClient, getMuleContainerVersion());

        List<URL> serverPlugins = getServerPlugins(mavenClient, containerConfiguration.getServerPlugins());
        ContainerInfo containerInfo = new ContainerInfo(muleVersion, containerBaseFolder, services, serverPlugins);

        startExecutorService();

        embeddedController = getEmbeddedController(embeddedControllerBootstrapClassLoader, containerInfo);
      } catch (Exception e) {
        // Clean up resources by calling stop in case of error during start
        try {
          doStop();
        } catch (Exception eStop) {
          e.addSuppressed(eStop);
        }
        throw new IllegalStateException("Cannot start embedded container", e);
      }
      started = true;
      startEmbeddedController(embeddedController);
    } else {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Embedded container already started");
      }
    }
  }

  @Override
  public synchronized void stop() {
    if (started) {
      try {
        doStop();
      } finally {
        started = false;
      }
    } else {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Embedded container already stopped");
      }
    }
  }

  private void doStop() {
    if (embeddedController != null) {
      stopEmbeddedController(embeddedController);
      embeddedController = null;
    }

    try {
      mavenClient.close();
    } catch (Exception e) {
      LOGGER.error("Error while closing 'mavenClient'", e);
    }

    stopExecutorService();

    closeClassLoader(containerModulesClassLoader);
    containerModulesClassLoader = null;
    closeClassLoader(embeddedControllerBootstrapClassLoader);
    embeddedControllerBootstrapClassLoader = null;
  }

  // The product may not be MuleRuntime, but something that contains it. Since the embedded-impl follows the version
  // schema of the runtime, we have to obtain it instead of assuming it is the same as the product's.
  @Override
  public String getMuleContainerVersion() {
    return callMethodFromContainerManifest("getProductVersion");
  }

  @Override
  public DeploymentService getDeploymentService() {
    return doGetDeploymentService(embeddedController);
  }

  @Override
  public File getContainerFolder() {
    return containerConfiguration.getContainerFolder();
  }

  @Override
  public boolean isCurrentJvmVersionSupported() {
    return parse(callMethodFromContainerManifest("getSupportedJdks"))
        .includes(getProperty("java.version"));
  }

  @Override
  public boolean isCurrentJvmVersionRecommended() {
    return parse(callMethodFromContainerManifest("getRecommndedJdks"))
        .includes(getProperty("java.version"));
  }

  private String callMethodFromContainerManifest(final String methodName) {
    ClassLoader originalTccl = currentThread().getContextClassLoader();
    currentThread().setContextClassLoader(containerModulesClassLoader);
    try {
      Class<?> clsMuleManifest = containerModulesClassLoader.loadClass("org.mule.runtime.core.api.config.MuleManifest");
      return (String) clsMuleManifest.getDeclaredMethod(methodName).invoke(null);
    } catch (ClassNotFoundException | IllegalAccessException | IllegalArgumentException | InvocationTargetException
        | NoSuchMethodException | SecurityException e) {
      throw new RuntimeException(e);
    } finally {
      currentThread().setContextClassLoader(originalTccl);
    }
  }

  protected abstract void startEmbeddedController(T embeddedController);

  protected abstract void stopEmbeddedController(T embeddedController);

  protected abstract DeploymentService doGetDeploymentService(T embeddedController);

  protected abstract T getEmbeddedController(ClassLoader embeddedControllerBootstrapClassLoader, ContainerInfo containerInfo)
      throws Exception;

  private ClassLoader createEmbeddedImplClassLoader(ClassLoader parentClassLoader, MavenClient mavenClient,
                                                    String muleVersion)
      throws MalformedURLException {
    BundleDescriptor embeddedBomDescriptor = new BundleDescriptor.Builder().setGroupId("org.mule.distributions")
        .setArtifactId("mule-module-embedded-impl-bom").setVersion(muleVersion).setType("pom").build();

    BundleDescriptor embeddedImplDescriptor = new BundleDescriptor.Builder().setGroupId("org.mule.distributions")
        .setArtifactId("mule-module-embedded-impl").setVersion(muleVersion).setType("jar").build();

    BundleDependency embeddedBundleImplDescriptor = mavenClient.resolveBundleDescriptor(embeddedImplDescriptor);

    List<BundleDependency> embeddedImplDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, embeddedBomDescriptor);

    List<URL> embeddedUrls = embeddedImplDependencies.stream()
        .filter(bundleDependency -> !bundleDependency.getScope().equals(PROVIDED))
        .map(dep -> {
          try {
            return dep.getBundleUri().toURL();
          } catch (MalformedURLException e) {
            throw new RuntimeException(e);
          }
        })
        .collect(toList());
    embeddedUrls.add(embeddedBundleImplDescriptor.getBundleUri().toURL());

    return new URLClassLoader(embeddedUrls.toArray(new URL[embeddedUrls.size()]), parentClassLoader);
  }

  private List<URL> getServerPlugins(MavenClient mavenClient, List<BundleDescriptor> serverPlugins) {
    return serverPlugins.stream().map(serverPluginDescriptor -> {
      BundleDependency bundleDependency = mavenClient.resolveBundleDescriptor(serverPluginDescriptor);
      try {
        return bundleDependency.getBundleUri().toURL();
      } catch (MalformedURLException e) {
        throw new RuntimeException(e);
      }
    }).collect(toList());
  }

  private void startExecutorService() {
    executorService = newSingleThreadExecutor(runnable -> new Thread(runnable, "Embedded"));
  }

  private void stopExecutorService() {
    if (executorService != null) {
      executorService.shutdownNow();
      executorService = null;
    }
  }

  private void closeClassLoader(ClassLoader classLoader) {
    if (classLoader instanceof Closeable) {
      try {
        ((Closeable) classLoader).close();
      } catch (IOException e) {
        // Do nothing.
        if (LOGGER.isDebugEnabled()) {
          LOGGER.debug("Failure closing container classloader", e);
        }
      }
    }
  }

  protected void executeUsingExecutorService(Callable<Void> callable) {
    Future<?> future = executorService.submit(() -> {
      ClassLoader contextClassLoader = currentThread().getContextClassLoader();
      try {
        currentThread().setContextClassLoader(containerModulesClassLoader);

        callable.call();
      } catch (Exception e) {
        throw new IllegalStateException(e);
      } finally {
        currentThread().setContextClassLoader(contextClassLoader);
      }
    });
    try {
      future.get();
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

}
