/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.embedded.internal;

import static java.util.Arrays.asList;
import static java.util.Collections.emptySet;
import static java.util.stream.Collectors.toList;

import static org.apache.commons.io.FileUtils.toFile;
import static org.apache.commons.lang3.JavaVersion.JAVA_11;
import static org.apache.commons.lang3.SystemUtils.IS_JAVA_1_8;
import static org.apache.commons.lang3.SystemUtils.isJavaVersionAtMost;

import org.mule.maven.client.api.BundleDependenciesResolutionException;
import org.mule.maven.client.api.MavenClient;
import org.mule.maven.pom.parser.api.model.BundleDependency;
import org.mule.maven.pom.parser.api.model.BundleDescriptor;
import org.mule.runtime.module.embedded.internal.classloading.JdkOnlyClassLoaderFactory;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Function;

/**
 * Creates a {@link ClassLoader} with the {@link URL}'s for the Container.
 *
 * @since 4.0
 */
// TODO MULE-11925 Improve MavenContainerClassLoaderFactory so it can work in terms of Dependency instead of URLs
public class MavenContainerClassLoaderFactory {

  private static final String CE_PATCHES_GROUP_ID = "org.mule.patches";
  private static final String EE_PATCHES_GROUP_ID = "com.mulesoft.mule.patches";

  private final MavenClient mavenClient;

  public MavenContainerClassLoaderFactory(MavenClient mavenClient) {
    this.mavenClient = mavenClient;
  }

  /**
   * Creates the Container {@link ClassLoader} for a given version.
   * 
   * @param runtimeProduct      mule product to bootstrap in embedded mode. Not null.
   * @param containerBaseFolder location of the container folder.
   * @return the Container {@link ClassLoader}.
   */
  public ClassLoader create(RuntimeProduct runtimeProduct, URL containerBaseFolder) {
    try {
      List<BundleDependency> bundleDependencies = resolveDependencies(runtimeProduct);

      List<URL> urls = bundleDependencies.stream()
          .map(dependencyToUrl())
          .collect(toList());
      urls = new ArrayList<>(urls);
      File containerFolderFile = toFile(containerBaseFolder);
      // the URL has to be constructed this way since File.toURI().toURL() gets rid of the final slash
      urls.add(new URL(new File(containerFolderFile, "conf").toURI() + "/"));
      return new URLClassLoader(urls.toArray(new URL[urls.size()]), buildParentClassLoader(this.getClass().getClassLoader()));
    } catch (BundleDependenciesResolutionException e) {
      throw new IllegalArgumentException("Could not find embedded container bom artifact", e);
    } catch (MalformedURLException e) {
      throw new RuntimeException(e);
    }
  }

  protected final ClassLoader buildParentClassLoader(ClassLoader classLoader) {
    Set<String> additionalExportedBootPackages = emptySet();
    if (isJavaVersionAtMost(JAVA_11)) {
      additionalExportedBootPackages =
          new HashSet<>(asList("org.mule.runtime.module.embedded.internal.controller", "org.mule.runtime.module.embedded.api"));
    }

    return JdkOnlyClassLoaderFactory.create(classLoader, additionalExportedBootPackages);
  }

  protected final List<BundleDependency> resolveDependencies(RuntimeProduct runtimeProduct) {
    return mavenClient
        .resolveBundleDescriptorDependencies(false, runtimeProduct.getContainerBomBundleDescriptor())
        .stream()
        // The services filter is kept for Mule Runtime product versions prior to 4.6
        .filter(bundleDependency -> !bundleDependency.getDescriptor().getGroupId().equals("org.mule.services"))
        .filter(this::shouldKeepXmlApis)
        .filter(bundleDependency -> !bundleDependency.getDescriptor().getType().equals("pom"))
        .sorted((dependency1, dependency2) -> {
          if (isPatchDependency(dependency1)) {
            return -1;
          } else if (isPatchDependency(dependency2)) {
            return 1;
          } else {
            return 0;
          }
        }).collect(toList());
  }

  protected final boolean shouldKeepXmlApis(BundleDependency bundleDependency) {
    if (IS_JAVA_1_8) {
      return true;
    } else {
      BundleDescriptor bundleDescriptor = bundleDependency.getDescriptor();

      // Java 11+ already contains the packages we need from this jar (org.w3c.dom.*),
      // so it is removed to avoid conflicts with the classes provided by the jdk.
      return !(bundleDescriptor.getGroupId().equals("xml-apis") && bundleDescriptor.getArtifactId().equals("xml-apis"));
    }
  }

  public final List<URL> getServices(RuntimeProduct runtimeProduct) {
    BundleDescriptor servicesBomBundleDescriptor = runtimeProduct.getServicesBomBundleDescriptor();
    List<BundleDependency> servicesDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, servicesBomBundleDescriptor);
    List<URL> urls = servicesDependencies.stream()
        .map(dependencyToUrl())
        .collect(toList());
    return urls.stream().filter(this::isService).collect(toList());
  }

  protected final Function<? super BundleDependency, ? extends URL> dependencyToUrl() {
    return dep -> {
      try {
        return dep.getBundleUri().toURL();
      } catch (MalformedURLException e) {
        throw new RuntimeException(e);
      }
    };
  }

  private boolean isService(URL url) {
    String fileName = toFile(url).getPath().toLowerCase();
    return fileName.endsWith(".zip") || fileName.endsWith("-mule-service.jar");
  }

  private static boolean isPatchDependency(BundleDependency dependency) {
    String groupId = dependency.getDescriptor().getGroupId();
    return groupId.equals(CE_PATCHES_GROUP_ID) || groupId.equals(EE_PATCHES_GROUP_ID);
  }

  protected MavenClient getMavenClient() {
    return mavenClient;
  }

}
