/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.embedded.internal;

import static org.mule.runtime.jpms.api.JpmsUtils.createModuleLayerClassLoader;
import static org.mule.runtime.jpms.api.MultiLevelClassLoaderFactory.MULTI_LEVEL_URL_CLASSLOADER_FACTORY;

import static java.lang.ClassLoader.getSystemClassLoader;
import static java.util.Collections.singletonList;
import static java.util.Optional.of;
import static java.util.stream.Collectors.partitioningBy;
import static java.util.stream.Collectors.toList;

import static org.apache.commons.io.FileUtils.toFile;

import org.mule.maven.client.api.BundleDependenciesResolutionException;
import org.mule.maven.client.api.MavenClient;
import org.mule.maven.pom.parser.api.model.BundleDependency;
import org.mule.maven.pom.parser.api.model.BundleDescriptor;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.List;
import java.util.Map;
import java.util.Properties;

/**
 * Creates a {@link ClassLoader} with the {@link URL}'s for the Container.
 *
 * @since 1.5
 */
public class MavenContainerOptSeparateClassLoaderFactory extends MavenContainerClassLoaderFactory {

  private static final String ARTIFACT_INFO_PROPERTIES = "artifact-info.properties";

  private final boolean isolate;

  public MavenContainerOptSeparateClassLoaderFactory(MavenClient mavenClient, boolean isolate) {
    super(mavenClient);
    this.isolate = isolate;
  }

  @Override
  public ClassLoader create(RuntimeProduct runtimeProduct, URL containerBaseFolder) {
    try {
      List<BundleDependency> bundleDependencies = resolveDependencies(runtimeProduct);

      final Map<Boolean, List<BundleDependency>> partitionedDependencies = bundleDependencies.stream()
          .collect(partitioningBy(bundleDependency -> isMuleContainerGroupId(bundleDependency.getDescriptor().getGroupId())));

      List<URL> muleUrls = partitionedDependencies.get(true)
          .stream()
          .map(dependencyToUrl())
          .collect(toList());
      List<URL> optUrls = partitionedDependencies.get(false)
          .stream()
          .map(dependencyToUrl())
          .collect(toList());

      File containerFolderFile = toFile(containerBaseFolder);
      // the URL has to be constructed this way since File.toURI().toURL() gets rid of the final slash
      URL configurationFolderUrl = new URL(new File(containerFolderFile, "conf").toURI() + "/");

      ClassLoader referenceClassLoader = getReferenceClassLoader(runtimeProduct);
      URLClassLoader parentClassLoader = new URLClassLoader(singletonList(configurationFolderUrl).toArray(new URL[1]),
                                                            buildParentClassLoader(referenceClassLoader));
      return createModuleLayerClassLoader(optUrls.toArray(new URL[optUrls.size()]),
                                          muleUrls.toArray(new URL[muleUrls.size()]),
                                          MULTI_LEVEL_URL_CLASSLOADER_FACTORY,
                                          parentClassLoader,
                                          of(getReferenceClass(runtimeProduct, referenceClassLoader)));
    } catch (BundleDependenciesResolutionException e) {
      throw new IllegalArgumentException("Could not find embedded container bom artifact", e);
    } catch (MalformedURLException e) {
      throw new RuntimeException(e);
    }
  }

  private ClassLoader getReferenceClassLoader(RuntimeProduct runtimeProduct) {
    if (runtimeProduct.isSupportsIsolation() && isolate) {
      return createModuleLayerClassLoader(new URL[] {getEmbeddedCommons()}, getSystemClassLoader().getParent());
    }

    return this.getClass().getClassLoader();
  }

  private Class<?> getReferenceClass(RuntimeProduct runtimeProduct, ClassLoader referenceClassLoader) {
    if (runtimeProduct.isSupportsIsolation() && isolate) {
      return getEmbeddedCommonsClass(referenceClassLoader);
    }

    return this.getClass();
  }

  private Class<?> getEmbeddedCommonsClass(ClassLoader commonsClassLoader) {
    try {
      return commonsClassLoader.loadClass("org.mule.runtime.module.embedded.commons.api.ArtifactConfiguration");
    } catch (ClassNotFoundException e) {
      throw new RuntimeException(e);
    }
  }

  private URL getEmbeddedCommons() {
    BundleDescriptor embeddedCommonsBundleDescriptor = new BundleDescriptor.Builder()
        .setGroupId("org.mule.runtime")
        .setArtifactId("mule-embedded-commons")
        .setVersion(getImplementationVersion())
        .setType("jar")
        .build();
    return dependencyToUrl().apply(getMavenClient()
        .resolveBundleDescriptor(embeddedCommonsBundleDescriptor));
  }

  private String getImplementationVersion() {
    try {
      Properties artifactInfo = new Properties();
      artifactInfo.load(this.getClass().getClassLoader().getResourceAsStream(ARTIFACT_INFO_PROPERTIES));

      return artifactInfo.getProperty("artifact-version");
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  // Implementation note: this must be kept consistent with the equivalent logic in test-runner and the distro assemblies
  private boolean isMuleContainerGroupId(final String groupId) {
    return groupId.equals("org.mule.runtime")
        || groupId.equals("org.mule.sdk")
        || groupId.equals("org.mule.weave")
        || groupId.equals("org.mule.mvel")
        || groupId.equals("org.mule.commons")
        || groupId.equals("com.mulesoft.mule.runtime")
        || groupId.equals("com.mulesoft.mule.runtime.modules")
        || groupId.equals("com.mulesoft.anypoint");
  }

}
