/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.embedded.internal;

import static org.mule.runtime.module.embedded.internal.legacy.Serializer.serialize;

import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;

import org.mule.maven.client.api.MavenClient;
import org.mule.runtime.module.embedded.api.ArtifactConfiguration;
import org.mule.runtime.module.embedded.api.ContainerConfiguration;
import org.mule.runtime.module.embedded.api.ContainerInfo;
import org.mule.runtime.module.embedded.api.DeploymentService;
import org.mule.runtime.module.embedded.api.EmbeddedContainer;
import org.mule.runtime.module.embedded.commons.api.DeploymentConfiguration;

import java.io.Serializable;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.URL;
import java.util.List;

import org.apache.commons.io.output.ByteArrayOutputStream;

/**
 * Implementation of {@link EmbeddedContainer} that uses reflection.
 */
public class ReflectionEmbeddedContainer extends AbstractEmbeddedContainer<Object> {

  public ReflectionEmbeddedContainer(String muleVersion, ContainerConfiguration containerConfiguration,
                                     ClassLoader containerModulesClassLoader, List<URL> services, URL containerBaseFolder,
                                     MavenClient mavenClient) {
    super(muleVersion, containerConfiguration, containerModulesClassLoader, services, containerBaseFolder, mavenClient);
  }

  @Override
  protected Object getEmbeddedController(ClassLoader embeddedControllerBootstrapClassLoader, ContainerInfo containerInfo)
      throws Exception {
    Class<?> controllerClass =
        embeddedControllerBootstrapClassLoader.loadClass(getControllerClassName());
    Constructor<?> constructor = controllerClass.getConstructor(byte[].class);
    ByteArrayOutputStream containerOutputStream = new ByteArrayOutputStream(512);
    serialize(adaptContainerInfo(containerInfo), containerOutputStream);
    return constructor.newInstance(containerOutputStream.toByteArray());
  }

  protected String getControllerClassName() {
    return "org.mule.runtime.module.embedded.impl.IsolatedEmbeddedController";
  }

  protected Serializable adaptContainerInfo(ContainerInfo containerInfo) {
    return new org.mule.runtime.module.embedded.commons.api.ContainerInfo(containerInfo.getVersion(),
                                                                          containerInfo.getContainerBaseFolder(),
                                                                          containerInfo.getServices(),
                                                                          containerInfo.getServerPlugins());
  }

  protected Serializable adaptArtifactConfiguration(ArtifactConfiguration artifactConfiguration) {
    DeploymentConfiguration deploymentConfiguration = DeploymentConfiguration.builder()
        .lazyConnectionsEnabled(artifactConfiguration.getDeploymentConfiguration().lazyConnectionsEnabled())
        .lazyInitialization(artifactConfiguration.getDeploymentConfiguration().lazyInitializationEnabled())
        .xmlValidations(artifactConfiguration.getDeploymentConfiguration().xmlValidationsEnabled())
        .addArtifactAstToRegistry(artifactConfiguration.getDeploymentConfiguration().addArtifactAstToRegistry())
        .doNotAddToolingObjectsToRegistry(artifactConfiguration.getDeploymentConfiguration().doNotAddToolingObjectsToRegistry())
        .build();

    return org.mule.runtime.module.embedded.commons.api.ArtifactConfiguration.builder()
        .artifactLocation(artifactConfiguration.getArtifactLocation())
        .deploymentConfiguration(deploymentConfiguration)
        .build();
  }

  @Override
  protected void startEmbeddedController(Object embeddedController) {
    executeUsingExecutorService(() -> {
      executeUsingReflection(embeddedController, () -> embeddedController.getClass().getMethod("start"), emptyList());
      return null;
    });
  }

  @Override
  protected void stopEmbeddedController(Object embeddedController) {
    executeUsingExecutorService(() -> {
      executeUsingReflection(embeddedController, () -> embeddedController.getClass().getMethod("stop"), emptyList());
      return null;
    });
  }

  @Override
  public DeploymentService doGetDeploymentService(Object embeddedController) {
    return new DeploymentService() {

      @Override
      public void deployApplication(ArtifactConfiguration artifactConfiguration) {
        executeUsingExecutorService(() -> {
          executeUsingReflection(embeddedController, () -> findEmbeddedMethod(embeddedController, "deployApplication"),
                                 singletonList(adaptArtifactConfiguration(artifactConfiguration)));
          return null;
        });
      }

      @Override
      public void undeployApplication(String applicationName) {
        executeUsingExecutorService(() -> {
          executeUsingReflection(embeddedController, () -> findEmbeddedMethod(embeddedController, "undeployApplication"),
                                 singletonList(applicationName));
          return null;
        });
      }

      @Override
      public void deployDomain(ArtifactConfiguration artifactConfiguration) {
        executeUsingExecutorService(() -> {
          executeUsingReflection(embeddedController, () -> findEmbeddedMethod(embeddedController, "deployDomain"),
                                 singletonList(adaptArtifactConfiguration(artifactConfiguration)));
          return null;
        });
      }

      @Override
      public void undeployDomain(String domainName) {
        executeUsingExecutorService(() -> {
          executeUsingReflection(embeddedController, () -> findEmbeddedMethod(embeddedController, "undeployDomain"),
                                 singletonList(domainName));
          return null;
        });
      }
    };
  }

  private Method findEmbeddedMethod(Object embeddedController, String methodName) {
    Method[] methods = embeddedController.getClass().getMethods();
    for (Method method : methods) {
      if (method.getName().equals(methodName)) {
        return method;
      }
    }
    throw new RuntimeException(new NoSuchMethodException("Method " + methodName + " not found"));
  }

  private void executeUsingReflection(Object embeddedController, UncheckedSupplier<Method> methodSupplier,
                                      List<Serializable> args)
      throws Exception {
    try {
      Method method = methodSupplier.get();
      List<byte[]> arguments = args.stream().map(arg -> {
        ByteArrayOutputStream byteOutputStream = new ByteArrayOutputStream();
        serialize(arg, byteOutputStream);
        return byteOutputStream.toByteArray();
      }).collect(toList());

      method.invoke(embeddedController, arguments.toArray(new Object[arguments.size()]));
    } catch (InvocationTargetException e) {
      Throwable cause = e.getCause();
      if (cause instanceof RuntimeException) {
        throw (RuntimeException) cause;
      } else {
        throw new IllegalStateException(cause);
      }
    }
  }

  interface UncheckedSupplier<T> {

    T get() throws Exception;

  }

}
