/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.embedded.internal;

import static org.mule.runtime.module.embedded.api.Product.MULE_FRAMEWORK;

import org.mule.maven.pom.parser.api.model.BundleDescriptor;
import org.mule.runtime.module.embedded.api.Product;
import org.mule.runtime.module.embedded.api.Product.ArtifactCoordinates;

import com.vdurmont.semver4j.Semver;

/**
 * Wraps a {@link Product} to provide different functionality according to its version.
 *
 * @since 1.6
 */
public class RuntimeProduct {

  /**
   * The Mule Runtime version from which the new embedded implementation is supported.
   */
  private static final Semver SUPPORTS_NEW_IMPL_VERSION = new Semver("4.5.0");
  private static final Semver MULE_46_VERSION = new Semver("4.6.0");
  private static final Semver MULE_466_VERSION = new Semver("4.6.6");

  private final Product product;
  private final String version;
  private final boolean supportsNewImplementation;
  private final boolean is46OrGreater;
  private final boolean supportsIsolation;

  public RuntimeProduct(Product product, String version) {
    this.product = product;
    this.version = version;

    Semver versionWithoutSuffix = new Semver(version).withClearedSuffix();
    boolean isMuleFramework = product.equals(MULE_FRAMEWORK);

    supportsNewImplementation = isMuleFramework || SUPPORTS_NEW_IMPL_VERSION.isLowerThanOrEqualTo(versionWithoutSuffix);
    is46OrGreater = isMuleFramework || MULE_46_VERSION.isLowerThanOrEqualTo(versionWithoutSuffix);
    supportsIsolation = isMuleFramework || MULE_466_VERSION.isLowerThanOrEqualTo(versionWithoutSuffix);
  }

  public boolean isSupportsNewImplementation() {
    return supportsNewImplementation;
  }

  public boolean isSupportsControllerApi() {
    return is46OrGreater;
  }

  public boolean isSupportsIsolation() {
    return supportsIsolation;
  }

  public BundleDescriptor getContainerBomBundleDescriptor() {
    ArtifactCoordinates containerBomArtifactCoordinates = getContainerBomArtifactCoordinates();
    return new BundleDescriptor.Builder()
        .setGroupId(containerBomArtifactCoordinates.getGroupId())
        .setArtifactId(containerBomArtifactCoordinates.getArtifactId())
        .setVersion(version)
        .setType("pom").build();
  }

  private ArtifactCoordinates getContainerBomArtifactCoordinates() {
    return is46OrGreater ? product.getContainerBomArtifactCoordinates() : product.getDeprecatedBomArtifactCoordinates();
  }

  public BundleDescriptor getServicesBomBundleDescriptor() {
    ArtifactCoordinates servicesBomArtifactCoordinates = getServicesBomArtifactCoordinates();
    return new BundleDescriptor.Builder()
        .setGroupId(servicesBomArtifactCoordinates.getGroupId())
        .setArtifactId(servicesBomArtifactCoordinates.getArtifactId())
        .setVersion(version)
        .setType("pom").build();
  }

  private ArtifactCoordinates getServicesBomArtifactCoordinates() {
    return is46OrGreater ? product.getServicesBomArtifactCoordinates() : product.getDeprecatedBomArtifactCoordinates();
  }

}
