/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.embedded.internal;

import static org.mule.runtime.module.embedded.internal.utils.DependenciesUtils.dependencyToUrl;

import static java.lang.String.format;
import static java.nio.file.Files.exists;
import static java.nio.file.Files.isDirectory;
import static java.nio.file.Files.walk;
import static java.nio.file.Paths.get;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.partitioningBy;
import static java.util.stream.Collectors.toList;

import static org.apache.commons.lang3.JavaVersion.JAVA_17;
import static org.apache.commons.lang3.SystemUtils.IS_JAVA_1_8;
import static org.apache.commons.lang3.SystemUtils.isJavaVersionAtLeast;

import org.mule.maven.pom.parser.api.model.BundleDescriptor;
import org.mule.runtime.module.embedded.api.dependencies.DependencyResolver;
import org.mule.runtime.module.embedded.api.dependencies.MuleDependenciesResolver;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

public class LocalDistroMuleDependenciesResolver implements MuleDependenciesResolver {

  private static final String BOOT_DIR = "lib/boot";
  private static final String MULE_DIR = "lib/mule";
  private static final String OPT_DIR = "lib/opt";
  private static final String SERVICES_DIR = "services";
  private static final String SERVER_PLUGINS_DIR = "server-plugins";
  private static final String JAR_FILE_EXTENSION = ".jar";

  private static final List<String> EXCLUDED_OPT_LIBS_FROM_BOOT = asList("disruptor", "log4j");
  private static final String MULE_LIBS_PREFIX = "mule-module";

  private final Path localDistributionPath;
  private final List<BundleDescriptor> serverPlugins;
  private final DependencyResolver dependencyResolver;
  private final boolean sanitize;
  private List<URL> containerUrls;
  private List<URL> muleUrls;
  private List<URL> optUrls;
  private List<URL> servicesUrls;
  private List<URL> serverPluginsUrls;

  public LocalDistroMuleDependenciesResolver(Path localDistributionPath) {
    this(localDistributionPath, emptyList(), null, true);
  }

  public LocalDistroMuleDependenciesResolver(Path localDistributionPath, List<BundleDescriptor> serverPlugins,
                                             DependencyResolver dependencyResolver, boolean sanitize) {
    if (!isDirectory(localDistributionPath)) {
      throw new IllegalArgumentException(format("Local distribution path '%s' is not a directory", localDistributionPath));
    }

    this.localDistributionPath = localDistributionPath;
    this.serverPlugins = serverPlugins;
    this.dependencyResolver = dependencyResolver;
    this.sanitize = sanitize;
  }

  @Override
  public List<URL> resolveMuleLibs() {
    if (containerUrls == null) {
      resolveContainerUrls();
    }

    return muleUrls;
  }

  @Override
  public List<URL> resolveOptLibs() {
    if (containerUrls == null) {
      resolveContainerUrls();
    }

    return optUrls;
  }

  @Override
  public List<URL> resolveMuleServices() {
    if (servicesUrls == null) {
      servicesUrls = getDirectoriesFromPath(localDistributionPath.resolve(SERVICES_DIR));
    }

    return servicesUrls;
  }

  @Override
  public List<URL> resolveServerPlugins() {
    if (serverPluginsUrls == null) {
      serverPluginsUrls = getDirectoriesFromPath(localDistributionPath.resolve(SERVER_PLUGINS_DIR));
      serverPluginsUrls
          .addAll(serverPlugins.stream().map(dependencyResolver::resolveBundleDescriptor).map(dependencyToUrl())
              .collect(toList()));
    }

    return serverPluginsUrls;
  }

  private List<URL> getDirectoriesFromPath(Path path) {
    List<URL> directories = new ArrayList<>();
    try (Stream<Path> paths = walk(path, 1)) {
      paths.skip(1).filter(Files::isDirectory).forEach(serviceDir -> directories.add(pathToUrl(serviceDir)));
    } catch (IOException e) {
      throw new RuntimeException(e);
    }

    return directories;
  }

  private void resolveContainerUrls() {
    Map<Boolean, List<URL>> partitionedBootLibs =
        libsFromDirectory(localDistributionPath.resolve(BOOT_DIR)).stream().collect(partitioningBy(this::isMuleLib));
    muleUrls = libsFromDirectory(localDistributionPath.resolve(MULE_DIR));
    muleUrls.addAll(partitionedBootLibs.get(true));
    optUrls = libsFromDirectory(localDistributionPath.resolve(OPT_DIR));
    optUrls.addAll(partitionedBootLibs.get(false).stream()
        .filter(this::sanitizeOptDependencies)
        .collect(toList()));
    containerUrls = new ArrayList<>(muleUrls);
    containerUrls.addAll(optUrls);
  }

  private boolean sanitizeOptDependencies(URL jar) {
    if (isJavaVersionAtLeast(JAVA_17) && sanitize) {
      return EXCLUDED_OPT_LIBS_FROM_BOOT.stream().noneMatch(excludedLib -> jar.toString().contains(excludedLib));
    }

    return true;
  }

  private boolean isMuleLib(URL jar) {
    return get(urlToUri(jar)).getFileName().toString().startsWith(MULE_LIBS_PREFIX);
  }

  private List<URL> libsFromDirectory(Path directory) {
    if (!exists(directory) || !isDirectory(directory)) {
      throw new RuntimeException(format("Invalid distribution directory structure. Couldn't find '%s'", directory));
    }

    try (Stream<Path> paths = walk(directory)) {
      return paths.filter(Files::isRegularFile)
          .filter(this::shouldKeepXmlApis)
          .filter(path -> path.toString().endsWith(JAR_FILE_EXTENSION))
          .map(p -> {
            try {
              return p.toUri().toURL();
            } catch (MalformedURLException e) {
              throw new RuntimeException(e);
            }
          })
          .collect(toList());
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  private URL pathToUrl(Path path) {
    try {
      return path.toUri().toURL();
    } catch (MalformedURLException e) {
      throw new RuntimeException(e);
    }
  }

  private URI urlToUri(URL url) {
    try {
      return url.toURI();
    } catch (URISyntaxException e) {
      throw new RuntimeException(e);
    }
  }

  private boolean shouldKeepXmlApis(Path jar) {
    if (IS_JAVA_1_8) {
      return true;
    } else {
      // Java 11+ already contains the packages we need from this jar (org.w3c.dom.*),
      // so it is removed to avoid conflicts with the classes provided by the jdk.
      return !(jar.toString().contains("xml-apis"));
    }
  }

}
