/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.embedded.internal;

import static org.mule.runtime.jpms.api.JpmsUtils.createModuleLayerClassLoader;
import static org.mule.runtime.jpms.api.MultiLevelClassLoaderFactory.MULTI_LEVEL_URL_CLASSLOADER_FACTORY;
import static org.mule.runtime.module.embedded.internal.utils.DependenciesUtils.dependencyToUrl;

import static java.lang.ClassLoader.getSystemClassLoader;
import static java.util.Collections.singletonList;
import static java.util.Optional.of;

import static org.apache.commons.io.FileUtils.toFile;

import org.mule.maven.client.api.BundleDependenciesResolutionException;
import org.mule.maven.client.api.MavenClient;
import org.mule.maven.pom.parser.api.model.BundleDescriptor;
import org.mule.runtime.module.embedded.api.dependencies.MuleDependenciesResolver;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.Properties;

/**
 * Creates a {@link ClassLoader} with the {@link URL}'s for the Container.
 *
 * @since 1.5
 */
public class MavenContainerOptSeparateClassLoaderFactory extends MavenContainerClassLoaderFactory {

  private static final String ARTIFACT_INFO_PROPERTIES = "artifact-info.properties";

  private final MavenClient mavenClient;
  private final RuntimeProduct runtimeProduct;
  private final boolean isolate;

  public MavenContainerOptSeparateClassLoaderFactory(MuleDependenciesResolver muleDependenciesResolver, MavenClient mavenClient,
                                                     RuntimeProduct runtimeProduct, boolean isolate) {
    super(muleDependenciesResolver);
    this.mavenClient = mavenClient;
    this.runtimeProduct = runtimeProduct;
    this.isolate = isolate;
  }

  @Override
  public ClassLoader create(URL containerBaseFolder) {
    try {

      File containerFolderFile = toFile(containerBaseFolder);
      // the URL has to be constructed this way since File.toURI().toURL() gets rid of the final slash
      URL configurationFolderUrl = new URL(new File(containerFolderFile, "conf").toURI() + "/");

      ClassLoader referenceClassLoader = getReferenceClassLoader(runtimeProduct);
      URLClassLoader parentClassLoader = new URLClassLoader(singletonList(configurationFolderUrl).toArray(new URL[1]),
                                                            buildParentClassLoader(referenceClassLoader));
      return createModuleLayerClassLoader(getMuleDependenciesResolver().resolveOptLibs().toArray(new URL[0]),
                                          getMuleDependenciesResolver().resolveMuleLibs().toArray(new URL[0]),
                                          MULTI_LEVEL_URL_CLASSLOADER_FACTORY,
                                          parentClassLoader,
                                          of(getReferenceClass(runtimeProduct, referenceClassLoader)));
    } catch (BundleDependenciesResolutionException e) {
      throw new IllegalArgumentException("Could not find embedded container bom artifact", e);
    } catch (MalformedURLException e) {
      throw new RuntimeException(e);
    }
  }

  private ClassLoader getReferenceClassLoader(RuntimeProduct runtimeProduct) {
    if (runtimeProduct.isSupportsIsolation() && isolate) {
      return createModuleLayerClassLoader(new URL[] {getEmbeddedCommons()}, getSystemClassLoader().getParent());
    }

    return this.getClass().getClassLoader();
  }

  private Class<?> getReferenceClass(RuntimeProduct runtimeProduct, ClassLoader referenceClassLoader) {
    if (runtimeProduct.isSupportsIsolation() && isolate) {
      return getEmbeddedCommonsClass(referenceClassLoader);
    }

    return this.getClass();
  }

  private Class<?> getEmbeddedCommonsClass(ClassLoader commonsClassLoader) {
    try {
      return commonsClassLoader.loadClass("org.mule.runtime.module.embedded.commons.api.ArtifactConfiguration");
    } catch (ClassNotFoundException e) {
      throw new RuntimeException(e);
    }
  }

  private URL getEmbeddedCommons() {
    BundleDescriptor embeddedCommonsBundleDescriptor = new BundleDescriptor.Builder()
        .setGroupId("org.mule.runtime")
        .setArtifactId("mule-embedded-commons")
        .setVersion(getImplementationVersion())
        .setType("jar")
        .build();
    return dependencyToUrl().apply(mavenClient
        .resolveBundleDescriptor(embeddedCommonsBundleDescriptor));
  }

  private String getImplementationVersion() {
    try {
      Properties artifactInfo = new Properties();
      artifactInfo.load(this.getClass().getClassLoader().getResourceAsStream(ARTIFACT_INFO_PROPERTIES));

      return artifactInfo.getProperty("artifact-version");
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

}
