/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.embedded.api.dependencies;

import org.mule.maven.pom.parser.api.model.BundleDescriptor;
import org.mule.runtime.module.embedded.api.Product;
import org.mule.runtime.module.embedded.internal.LocalDistroMuleDependenciesResolver;
import org.mule.runtime.module.embedded.internal.MavenClientMuleDependenciesResolver;
import org.mule.runtime.module.embedded.internal.RuntimeProduct;

import java.net.URL;
import java.nio.file.Path;
import java.util.List;

/**
 * Resolution of Mule Runtime dependencies, i.e. its libraries and services.
 *
 * @since 1.8
 */
public interface MuleDependenciesResolver {

  /**
   * @param product               Mule Runtime {@link Product} to get the dependencies for. Only to be used in case
   *                              {@code localDistributionPath} is not provided.
   * @param version               Mule Runtime version to get the dependencies for. Only to be used in case
   *                              {@code localDistributionPath} is not provided.
   * @param serverPlugins         the server-plugins to be resolved. Only to be used in case {@code localDistributionPath} is not
   *                              provided.
   * @param dependencyResolver    the {@link DependencyResolver} to perform the dependency resolution. Only to be used in case
   *                              {@code localDistributionPath} is not provided.
   * @param localDistributionPath local distribution from where to fetch the dependencies.
   * @param sanitize              whether the dependencies should be sanitized as needed for an embedded context.
   * @return the default implementation of a {@link MuleDependenciesResolver}.
   */
  static MuleDependenciesResolver getMuleDependenciesResolver(Product product,
                                                              String version,
                                                              List<BundleDescriptor> serverPlugins,
                                                              DependencyResolver dependencyResolver,
                                                              Path localDistributionPath,
                                                              boolean sanitize) {
    RuntimeProduct runtimeProduct = new RuntimeProduct(product, version);
    return localDistributionPath != null
        ? new LocalDistroMuleDependenciesResolver(localDistributionPath, product, serverPlugins, dependencyResolver, sanitize)
        : new MavenClientMuleDependenciesResolver(runtimeProduct.getContainerBomBundleDescriptor(),
                                                  runtimeProduct.getServicesBomBundleDescriptor(),
                                                  serverPlugins,
                                                  dependencyResolver,
                                                  sanitize);
  }

  /**
   * @return the Mule Runtime libraries for the container.
   */
  List<URL> resolveMuleLibs();

  /**
   * @return the third party and non-core libraries for the container.
   */
  List<URL> resolveOptLibs();

  /**
   * @return the Mule Runtime services for the container.
   */
  List<URL> resolveMuleServices();

  /**
   * @return the Mule Runtime server plugins for the container.
   */
  List<URL> resolveServerPlugins();

}
