/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.embedded.internal;

import static java.util.Arrays.asList;
import static java.util.Collections.unmodifiableList;
import static java.util.Objects.requireNonNull;

import org.mule.maven.pom.parser.api.model.BundleDescriptor;
import org.mule.runtime.module.embedded.api.ContainerConfiguration;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

/**
 * Default implementation of {@link ContainerConfiguration.ContainerConfigurationBuilder}
 * 
 * @since 1.0
 */
public class DefaultContainerConfigurationBuilder implements ContainerConfiguration.ContainerConfigurationBuilder {

  private File containerFolder;
  private List<BundleDescriptor> bundleDescriptors = new ArrayList<>();

  @Override
  public ContainerConfiguration.ContainerConfigurationBuilder containerFolder(File containerFolder) {
    this.containerFolder = containerFolder;
    return this;
  }

  @Override
  public ContainerConfiguration.ContainerConfigurationBuilder serverPlugins(BundleDescriptor... bundleDescriptors) {
    this.bundleDescriptors = asList(bundleDescriptors);
    return this;
  }

  @Override
  public ContainerConfiguration.ContainerConfigurationBuilder serverPlugins(List<BundleDescriptor> bundleDescriptors) {
    this.bundleDescriptors = unmodifiableList(bundleDescriptors);
    return this;
  }

  @Override
  public ContainerConfiguration build() {
    return new ContainerConfigurationImpl(containerFolder, bundleDescriptors);
  }

  public static class ContainerConfigurationImpl implements ContainerConfiguration {

    private static final long serialVersionUID = -560574823186446469L;

    private final File containerFolder;
    private final List<BundleDescriptor> serverPlugins;

    public ContainerConfigurationImpl(File containerFolder, List<BundleDescriptor> serverPlugins) {
      requireNonNull(containerFolder, "containerFolder cannot be null");
      requireNonNull(serverPlugins, "serverPlugins cannot be null");
      this.containerFolder = containerFolder;
      this.serverPlugins = serverPlugins;
    }

    @Override
    public File getContainerFolder() {
      return this.containerFolder;
    }

    @Override
    public List<BundleDescriptor> getServerPlugins() {
      return this.serverPlugins;
    }
  }


}
