/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.embedded.internal;

import static java.util.Arrays.asList;
import static java.util.Collections.emptySet;

import static org.apache.commons.io.FileUtils.toFile;
import static org.apache.commons.lang3.JavaVersion.JAVA_11;
import static org.apache.commons.lang3.SystemUtils.isJavaVersionAtMost;

import org.mule.maven.client.api.BundleDependenciesResolutionException;
import org.mule.runtime.module.embedded.internal.classloading.JdkOnlyClassLoaderFactory;
import org.mule.runtime.module.embedded.api.dependencies.MuleDependenciesResolver;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * Creates a {@link ClassLoader} with the {@link URL}'s for the Container.
 *
 * @since 4.0
 */
// TODO MULE-11925 Improve MavenContainerClassLoaderFactory so it can work in terms of Dependency instead of URLs
public class MavenContainerClassLoaderFactory {

  private final MuleDependenciesResolver muleDependenciesResolver;

  public MavenContainerClassLoaderFactory(MuleDependenciesResolver muleDependenciesResolver) {
    this.muleDependenciesResolver = muleDependenciesResolver;
  }

  /**
   * Creates the Container {@link ClassLoader} for a given version.
   *
   * @param containerBaseFolder location of the container folder.
   * @return the Container {@link ClassLoader}.
   */
  public ClassLoader create(URL containerBaseFolder) {
    try {
      List<URL> urls = resolveDependencies(muleDependenciesResolver);
      File containerFolderFile = toFile(containerBaseFolder);
      // the URL has to be constructed this way since File.toURI().toURL() gets rid of the final slash
      urls.add(new URL(new File(containerFolderFile, "conf").toURI() + "/"));
      return new URLClassLoader(urls.toArray(new URL[urls.size()]), buildParentClassLoader(this.getClass().getClassLoader()));
    } catch (BundleDependenciesResolutionException e) {
      throw new IllegalArgumentException("Could not find embedded container bom artifact", e);
    } catch (MalformedURLException e) {
      throw new RuntimeException(e);
    }
  }

  private List<URL> resolveDependencies(MuleDependenciesResolver muleDependenciesResolver) {
    List<URL> urls = new ArrayList<>(muleDependenciesResolver.resolveOptLibs());
    urls.addAll(muleDependenciesResolver.resolveMuleLibs());

    return urls;
  }

  protected final ClassLoader buildParentClassLoader(ClassLoader classLoader) {
    Set<String> additionalExportedBootPackages = emptySet();
    if (isJavaVersionAtMost(JAVA_11)) {
      additionalExportedBootPackages =
          new HashSet<>(asList("org.mule.runtime.module.embedded.internal.controller", "org.mule.runtime.module.embedded.api"));
    }

    return JdkOnlyClassLoaderFactory.create(classLoader, additionalExportedBootPackages);
  }

  protected final MuleDependenciesResolver getMuleDependenciesResolver() {
    return muleDependenciesResolver;
  }

}
