/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.embedded.internal;

import static org.mule.maven.client.api.MavenClientProvider.discoverProvider;
import static org.mule.runtime.module.embedded.api.dependencies.MuleDependenciesResolver.getMuleDependenciesResolver;
import static org.mule.runtime.module.embedded.internal.utils.JpmsUtils.validateNoBootModuleLayerTweaking;
import static org.mule.runtime.module.embedded.internal.utils.Preconditions.checkState;

import static java.lang.Thread.currentThread;
import static java.nio.charset.StandardCharsets.UTF_8;
import static java.nio.file.Paths.get;

import static org.apache.commons.io.FileUtils.write;

import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.MavenClientProvider;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.runtime.module.embedded.api.ContainerConfiguration;
import org.mule.runtime.module.embedded.api.EmbeddedContainer;
import org.mule.runtime.module.embedded.api.EmbeddedContainer.EmbeddedContainerBuilder;
import org.mule.runtime.module.embedded.api.Product;
import org.mule.runtime.module.embedded.api.dependencies.MuleDependenciesResolver;
import org.mule.runtime.module.embedded.internal.dependencies.DefaultDependencyResolver;
import org.mule.runtime.module.embedded.internal.legacy.LegacyDefaultEmbeddedContainer;
import org.mule.runtime.module.embedded.internal.legacy.LegacyEmbeddedContainer;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Path;
import java.util.List;

import com.google.gson.Gson;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;

public class DefaultEmbeddedContainerBuilder implements EmbeddedContainerBuilder {

  private String muleVersion;
  private MavenConfiguration mavenConfiguration;
  private ContainerConfiguration containerConfiguration;
  private Product product;
  private boolean isolate = true;
  private Path localDistributionPath;

  @Override
  public EmbeddedContainerBuilder muleVersion(String muleVersion) {
    this.muleVersion = muleVersion;
    return this;
  }

  @Override
  public EmbeddedContainerBuilder product(Product product) {
    this.product = product;
    return this;
  }

  @Override
  public EmbeddedContainerBuilder containerConfiguration(ContainerConfiguration containerConfiguration) {
    this.containerConfiguration = containerConfiguration;
    return this;
  }

  @Override
  @Deprecated
  public EmbeddedContainerBuilder log4jConfigurationFile(URI log4JConfigurationFile) {
    return this;
  }

  @Override
  public EmbeddedContainerBuilder mavenConfiguration(MavenConfiguration mavenConfiguration) {
    this.mavenConfiguration = mavenConfiguration;
    return this;
  }

  @Override
  public EmbeddedContainerBuilder useIsolation(boolean isolate) {
    this.isolate = isolate;
    return this;
  }

  @Override
  public EmbeddedContainerBuilder fromLocalDistribution(Path localDistributionPath) {
    this.localDistributionPath = localDistributionPath;
    return this;
  }

  @Override
  public EmbeddedContainer build() {
    checkState(muleVersion != null, "muleVersion cannot be null");
    checkState(mavenConfiguration != null, "mavenConfiguration cannot be null");
    checkState(containerConfiguration != null, "containerConfiguration cannot be null");
    checkState(product != null, "product cannot be null");

    validateNoBootModuleLayerTweaking();

    try {
      URL containerBaseFolder = containerConfiguration.getContainerFolder().toURI().toURL();

      RuntimeProduct runtimeProduct = new RuntimeProduct(product, muleVersion);

      final ClassLoader tccl = currentThread().getContextClassLoader();
      MavenClientProvider mavenClientProvider = discoverMavenClientProvider(tccl, getClass().getClassLoader());
      MavenClient mavenClient = mavenClientProvider.createMavenClient(mavenConfiguration);

      persistMavenConfiguration(containerBaseFolder, mavenConfiguration);

      MuleDependenciesResolver muleDependenciesResolver =
          getMuleDependenciesResolver(product, muleVersion, containerConfiguration.getServerPlugins(),
                                      new DefaultDependencyResolver(mavenClient), localDistributionPath, true);

      MavenContainerClassLoaderFactory classLoaderFactory = runtimeProduct.isSupportsNewImplementation()
          ? new MavenContainerOptSeparateClassLoaderFactory(muleDependenciesResolver, mavenClient, runtimeProduct, isolate)
          : new MavenContainerClassLoaderFactory(muleDependenciesResolver);

      try {
        ClassLoader containerModulesClassLoader = classLoaderFactory.create(containerBaseFolder.toURI().toURL());

        return getEmbeddedContainer(runtimeProduct, containerBaseFolder, mavenClient,
                                    muleDependenciesResolver.resolveMuleServices(),
                                    muleDependenciesResolver.resolveServerPlugins(), containerModulesClassLoader);
      } catch (Exception e) {
        throw new IllegalStateException("Cannot create embedded container", e);
      }
    } catch (Exception e) {
      if (e instanceof RuntimeException) {
        throw (RuntimeException) e;
      } else {
        throw new RuntimeException(e);
      }
    }
  }

  public static MavenClientProvider discoverMavenClientProvider(final ClassLoader tccl, final ClassLoader fallbackCl) {
    if (tccl == null) {
      return discoverProvider(fallbackCl);
    }

    try {
      return discoverProvider(tccl);
    } catch (IllegalStateException e) {
      // fallback
      return discoverProvider(fallbackCl);
    }
  }

  private EmbeddedContainer getEmbeddedContainer(RuntimeProduct runtimeProduct, URL containerBaseFolder, MavenClient mavenClient,
                                                 List<URL> services, List<URL> serverPlugins,
                                                 ClassLoader containerModulesClassLoader) {
    if (runtimeProduct.isSupportsNewImplementation()) {
      if (runtimeProduct.isSupportsIsolation()) {
        return new ReflectionEmbeddedContainer(muleVersion, containerConfiguration, containerModulesClassLoader,
                                               services, serverPlugins, containerBaseFolder, mavenClient);
      } else if (runtimeProduct.isSupportsControllerApi()) {
        return new DefaultEmbeddedContainer(muleVersion, containerConfiguration, containerModulesClassLoader,
                                            services, serverPlugins, containerBaseFolder, mavenClient);
      } else {
        return new LegacyDefaultEmbeddedContainer(muleVersion, containerConfiguration, containerModulesClassLoader,
                                                  services, serverPlugins, containerBaseFolder,
                                                  mavenClient);
      }
    } else {
      return new LegacyEmbeddedContainer(muleVersion, containerConfiguration, containerModulesClassLoader,
                                         services, serverPlugins, containerBaseFolder, mavenClient);
    }
  }

  private void persistMavenConfiguration(URL containerBaseFolder, MavenConfiguration mavenConfiguration)
      throws IOException, URISyntaxException {
    File configurationFolder = new File(get(containerBaseFolder.toURI()).toFile(), "conf");
    if (!configurationFolder.exists()) {
      if (!configurationFolder.mkdirs()) {
        throw new IllegalArgumentException("Could not create MULE_HOME/conf folder in: " + configurationFolder.getAbsolutePath());
      }
    }

    JsonObject rootObject = new JsonObject();
    JsonObject muleRuntimeConfigObject = new JsonObject();
    rootObject.add("muleRuntimeConfig", muleRuntimeConfigObject);
    JsonObject mavenObject = new JsonObject();
    muleRuntimeConfigObject.add("maven", mavenObject);
    if (!mavenConfiguration.getMavenRemoteRepositories().isEmpty()) {
      JsonObject repositoriesObject = new JsonObject();
      mavenObject.add("repositories", repositoriesObject);
      mavenConfiguration.getMavenRemoteRepositories().forEach(mavenRepo -> {
        JsonObject repoObject = new JsonObject();
        repositoriesObject.add(mavenRepo.getId(), repoObject);
        repoObject.addProperty("url", mavenRepo.getUrl().toString());
        mavenRepo.getAuthentication().ifPresent(authentication -> {
          repoObject.addProperty("username", authentication.getUsername());
          repoObject.addProperty("password", authentication.getPassword());
        });
        mavenRepo.getSnapshotPolicy().ifPresent(snapshotPolicy -> {
          JsonObject snapshotPolicyObject = new JsonObject();
          snapshotPolicyObject.addProperty("enabled", snapshotPolicy.isEnabled());
          snapshotPolicyObject.addProperty("updatePolicy", snapshotPolicy.getUpdatePolicy());
          snapshotPolicyObject.addProperty("checksumPolicy", snapshotPolicy.getChecksumPolicy());
          repoObject.add("snapshotPolicy", snapshotPolicyObject);
        });
        mavenRepo.getReleasePolicy().ifPresent(releasePolicy -> {
          JsonObject releasePolicyObject = new JsonObject();
          releasePolicyObject.addProperty("enabled", releasePolicy.isEnabled());
          releasePolicyObject.addProperty("updatePolicy", releasePolicy.getUpdatePolicy());
          releasePolicyObject.addProperty("checksumPolicy", releasePolicy.getChecksumPolicy());
          repoObject.add("releasePolicy", releasePolicyObject);
        });
      });
    }
    mavenObject.addProperty("repositoryLocation", mavenConfiguration.getLocalMavenRepositoryLocation().getAbsolutePath());
    mavenConfiguration.getUserSettingsLocation().ifPresent(userSettingsLocation -> mavenObject
        .addProperty("userSettingsLocation", userSettingsLocation.getAbsolutePath()));
    mavenConfiguration.getGlobalSettingsLocation().ifPresent(globalSettingsLocation -> mavenObject
        .addProperty("globalSettingsLocation", globalSettingsLocation.getAbsolutePath()));
    mavenObject.addProperty("ignoreArtifactDescriptorRepositories", mavenConfiguration.getIgnoreArtifactDescriptorRepositories());
    mavenConfiguration.getSettingsSecurityLocation().ifPresent(securitySettingsLocation -> mavenObject
        .addProperty("settingsSecurityLocation", securitySettingsLocation.getAbsolutePath()));

    mavenObject.addProperty("forcePolicyUpdateNever", mavenConfiguration.getForcePolicyUpdateNever());
    mavenObject.addProperty("forcePolicyUpdateAlways", mavenConfiguration.getForcePolicyUpdateAlways());

    mavenConfiguration.getActiveProfiles().ifPresent(activeProfiles -> {
      JsonArray activeProfilesArray = new JsonArray();
      activeProfiles.forEach(activeProfilesArray::add);
      mavenObject.add("activeProfiles", activeProfilesArray);
    });
    mavenConfiguration.getInactiveProfiles().ifPresent(inactiveProfiles -> {
      JsonArray inactiveProfilesArray = new JsonArray();
      inactiveProfiles.forEach(inactiveProfilesArray::add);
      mavenObject.add("inactiveProfiles", inactiveProfilesArray);
    });

    mavenObject.addProperty("offLineMode", mavenConfiguration.getOfflineMode());

    mavenConfiguration.getUserProperties().ifPresent(userProperties -> {
      JsonObject userPropertiesObject = new JsonObject();
      mavenObject.add("userProperties", userPropertiesObject);
      userProperties.entrySet().forEach(userPropertyEntry -> {
        JsonObject userPropertyObject = new JsonObject();
        userPropertiesObject.add(userPropertyEntry.getKey().toString(), userPropertyObject);
        userPropertyObject.addProperty("value", userPropertyEntry.getValue().toString());
      });
    });

    String muleConfigContent = new Gson().toJson(rootObject);
    write(new File(configurationFolder, "mule-config.json"), muleConfigContent, UTF_8);
  }

}
