/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.embedded.internal.jvm;

import static com.vdurmont.semver4j.Range.RangeOperator.GT;
import static com.vdurmont.semver4j.Range.RangeOperator.GTE;
import static com.vdurmont.semver4j.Range.RangeOperator.LT;
import static com.vdurmont.semver4j.Range.RangeOperator.LTE;
import static com.vdurmont.semver4j.Semver.SemverType.LOOSE;

import com.vdurmont.semver4j.Range;
import com.vdurmont.semver4j.Semver;

/**
 * Represents an interval of supported Java versions.
 * <p>
 * Implementation is immutable and thread-safe.
 * 
 * @since 1.6
 */
public class JvmVersionInterval {

  /**
   * Parses an intervals of supported Java versions. Endpoints may be open or closed, indicated by parenthesis or brackets
   * respectively.
   * <p>
   * Example: {@code [1.8.0_181,9)}
   * 
   * @param intervalString the string to parse.
   * @return the parsed JvmVersionInterval
   */
  public static JvmVersionInterval parse(String intervalString) {
    final String[] intervalEndpoints = intervalString.split(",");
    String lowerEndpoint = intervalEndpoints[0];
    String higherEndpoint = intervalEndpoints[1];

    boolean lowerOpen;
    if (lowerEndpoint.startsWith("(")) {
      lowerOpen = true;
    } else if (lowerEndpoint.startsWith("[")) {
      lowerOpen = false;
    } else {
      throw new IllegalArgumentException("Invalid open/closed notation ('" + lowerEndpoint.charAt(0) + "') for lower endpoint '"
          + lowerEndpoint + "'. '(' or '[' expected.");
    }

    boolean higherOpen;
    if (higherEndpoint.endsWith(")")) {
      higherOpen = true;
    } else if (higherEndpoint.endsWith("]")) {
      higherOpen = false;
    } else {
      throw new IllegalArgumentException("Invalid open/closed notation ('" + higherEndpoint.charAt(higherEndpoint.length() - 1)
          + "') for higher endpoint '" + higherEndpoint + "'. '(' or '[' expected.");
    }

    return new JvmVersionInterval(lowerEndpoint.substring(1),
                                  lowerOpen,
                                  higherEndpoint.substring(0, higherEndpoint.length() - 1),
                                  higherOpen);
  }

  private final Range lowerRange;
  private final Range higherRange;

  /**
   * Creates a {@link JvmVersionInterval} with the provided parameters.
   */
  public JvmVersionInterval(String lowerEndpoint, boolean lowerOpen, String higherEndpoint, boolean higherOpen) {
    this.lowerRange = new Range(new Semver(normalizeJava8OrLowerVersion(lowerEndpoint), LOOSE).toStrict(),
                                lowerOpen ? GT : GTE);
    this.higherRange = new Range(new Semver(normalizeJava8OrLowerVersion(higherEndpoint), LOOSE).toStrict(),
                                 higherOpen ? LT : LTE);
  }

  /**
   * Checks if this interval includes the version indicated by {@code versionString}.
   * 
   * @param versionString the version to check for inclusion.
   * @return {@code true} if this interval contains the version indicated by {@code versionString}, {@code false} otherwise.
   */
  public boolean includes(Semver version) {
    return lowerRange.isSatisfiedBy(version) && higherRange.isSatisfiedBy(version);
  }

  @Override
  public String toString() {
    return lowerRange.toString() + " && " + higherRange.toString();
  }

  /**
   * <ul>
   * <li>Remove the 1. prefix from pre java 8 versions so it can be interpreted as semver.</li>
   * <li>Replace `_` with `.` for proper semver parsing.</li>
   * </ul>
   */
  static String normalizeJava8OrLowerVersion(String versionString) {
    if (versionString.startsWith("1.")) {
      return versionString.substring(2).replace("_", ".");
    } else {
      return versionString.replace("_", ".");
    }
  }

}
