/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.embedded.internal;

import static org.mule.runtime.module.embedded.internal.utils.DependenciesUtils.dependencyToUrl;
import static org.mule.runtime.module.embedded.internal.utils.Preconditions.checkArgument;

import static java.util.stream.Collectors.partitioningBy;
import static java.util.stream.Collectors.toList;

import static org.apache.commons.lang3.JavaVersion.JAVA_17;
import static org.apache.commons.lang3.SystemUtils.IS_JAVA_1_8;
import static org.apache.commons.lang3.SystemUtils.isJavaVersionAtLeast;

import org.mule.maven.client.api.MavenClient;
import org.mule.maven.pom.parser.api.model.BundleDependency;
import org.mule.maven.pom.parser.api.model.BundleDescriptor;
import org.mule.runtime.module.embedded.api.dependencies.DependencyResolver;
import org.mule.runtime.module.embedded.api.dependencies.MuleDependenciesResolver;

import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * Resolves Mule dependencies based on the {@link MavenClient}.
 */
public class MavenClientMuleDependenciesResolver implements MuleDependenciesResolver {

  private static final Collection<ExcludedDependency> EXCLUDED_DEPENDENCIES;

  static {
    EXCLUDED_DEPENDENCIES = new ArrayList<>();
    EXCLUDED_DEPENDENCIES.add(new ExcludedDependency("org.apache.logging.log4j"));
    EXCLUDED_DEPENDENCIES.add(new ExcludedDependency("com.lmax", "disruptor"));
  }

  private static final String CE_PATCHES_GROUP_ID = "org.mule.patches";
  private static final String EE_PATCHES_GROUP_ID = "com.mulesoft.mule.patches";

  private final BundleDescriptor muleLibsBom;
  private final BundleDescriptor muleServicesBom;
  private final List<BundleDescriptor> serverPlugins;
  private final DependencyResolver dependencyResolver;
  private final boolean sanitize;
  private Map<Boolean, List<BundleDependency>> containerDependencies;
  private List<URL> muleUrls;
  private List<URL> optUrls;
  private List<URL> servicesUrls;
  private List<URL> serverPluginsUrls;

  public MavenClientMuleDependenciesResolver(BundleDescriptor muleLibsBom,
                                             BundleDescriptor muleServicesBom,
                                             List<BundleDescriptor> serverPlugins,
                                             DependencyResolver dependencyResolver,
                                             boolean sanitize) {
    this.muleLibsBom = muleLibsBom;
    this.muleServicesBom = muleServicesBom;
    this.serverPlugins = serverPlugins;
    this.dependencyResolver = dependencyResolver;
    this.sanitize = sanitize;
  }

  @Override
  public List<URL> resolveMuleLibs() {
    if (containerDependencies == null) {
      resolveContainerDependencies();
    }

    return muleUrls;
  }

  @Override
  public List<URL> resolveOptLibs() {
    if (containerDependencies == null) {
      resolveContainerDependencies();
    }

    return optUrls;
  }

  private void resolveContainerDependencies() {
    containerDependencies = dependencyResolver
        .resolveBundleDescriptorDependencies(muleLibsBom)
        .stream()
        // The services filter is kept for Mule Runtime product versions prior to 4.6
        .filter(bundleDependency -> !bundleDependency.getDescriptor().getGroupId().equals("org.mule.services"))
        .filter(this::shouldKeepXmlApis)
        .filter(bundleDependency -> !bundleDependency.getDescriptor().getType().equals("pom"))
        .sorted((dependency1, dependency2) -> {
          if (isPatchDependency(dependency1)) {
            return -1;
          } else if (isPatchDependency(dependency2)) {
            return 1;
          } else {
            return 0;
          }
        })
        .collect(partitioningBy(bundleDependency -> isMuleContainerGroupId(bundleDependency.getDescriptor().getGroupId())));

    muleUrls = containerDependencies.get(true)
        .stream()
        .map(dependencyToUrl())
        .collect(toList());
    optUrls = containerDependencies.get(false)
        .stream()
        .filter(this::sanitizeOptDependencies)
        .map(dependencyToUrl())
        .collect(toList());

  }

  @Override
  public List<URL> resolveMuleServices() {
    if (servicesUrls == null) {
      servicesUrls =
          dependencyResolver.resolveBundleDescriptorDependencies(muleServicesBom).stream().filter(this::isService)
              .map(dependencyToUrl()).collect(toList());
    }

    return servicesUrls;
  }

  @Override
  public List<URL> resolveServerPlugins() {
    if (serverPluginsUrls == null) {
      serverPluginsUrls =
          serverPlugins.stream().map(dependencyResolver::resolveBundleDescriptor).map(dependencyToUrl()).collect(toList());
    }

    return serverPluginsUrls;
  }

  // Implementation note: this must be kept consistent with the equivalent logic in test-runner and the distro assemblies
  private boolean isMuleContainerGroupId(final String groupId) {
    return groupId.equals("org.mule.runtime")
        || groupId.equals("org.mule.runtime.boot")
        || groupId.equals("org.mule.sdk")
        || groupId.equals("org.mule.weave")
        || groupId.equals("org.mule.mvel")
        || groupId.equals("org.mule.commons")
        || groupId.equals("com.mulesoft.mule.runtime")
        || groupId.equals("com.mulesoft.mule.runtime.boot")
        || groupId.equals("com.mulesoft.mule.runtime.modules")
        || groupId.equals("com.mulesoft.anypoint")
        || groupId.equals("com.mulesoft.connectivity");
  }

  private static boolean isPatchDependency(BundleDependency dependency) {
    String groupId = dependency.getDescriptor().getGroupId();
    return groupId.equals(CE_PATCHES_GROUP_ID) || groupId.equals(EE_PATCHES_GROUP_ID);
  }

  private boolean shouldKeepXmlApis(BundleDependency bundleDependency) {
    if (IS_JAVA_1_8) {
      return true;
    } else {
      BundleDescriptor bundleDescriptor = bundleDependency.getDescriptor();

      // Java 11+ already contains the packages we need from this jar (org.w3c.dom.*),
      // so it is removed to avoid conflicts with the classes provided by the jdk.
      return !(bundleDescriptor.getGroupId().equals("xml-apis") && bundleDescriptor.getArtifactId().equals("xml-apis"));
    }
  }

  private boolean isService(BundleDependency dependency) {
    return dependency.getDescriptor().getClassifier().map(classifier -> classifier.equals("mule-service")).orElse(false);
  }

  private boolean sanitizeOptDependencies(BundleDependency bundleDependency) {
    if (isJavaVersionAtLeast(JAVA_17) && sanitize) {
      return EXCLUDED_DEPENDENCIES.stream().noneMatch(excludedDependency -> excludedDependency.match(bundleDependency));
    }

    return true;
  }

  private static class ExcludedDependency {

    private final String groupId;
    private final String artifactId;

    public ExcludedDependency(String groupId) {
      this(groupId, null);
    }

    public ExcludedDependency(String groupId, String artifactId) {
      checkArgument(groupId != null, "'groupId' can't be null");
      this.groupId = groupId;
      this.artifactId = artifactId;
    }

    public boolean match(BundleDependency bundleDependency) {
      boolean match = groupId.equals(bundleDependency.getDescriptor().getGroupId());
      if (artifactId != null) {
        match = match && artifactId.equals(bundleDependency.getDescriptor().getArtifactId());
      }

      return match;
    }
  }

}
