/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.embedded.commons.internal;

import static org.mule.runtime.module.embedded.commons.internal.utils.Preconditions.checkArgument;

import org.mule.runtime.module.embedded.commons.api.ArtifactConfiguration;
import org.mule.runtime.module.embedded.commons.api.ArtifactConfiguration.ArtifactConfigurationBuilder;
import org.mule.runtime.module.embedded.commons.api.DeploymentConfiguration;

import java.io.File;

/**
 * Default implementation of {@link ArtifactConfigurationBuilder}.
 */
public class DefaultArtifactConfigurationBuilder implements ArtifactConfigurationBuilder {

  private File applicationLocation;
  private DeploymentConfiguration deploymentConfiguration = DeploymentConfiguration.builder().build();

  @Override
  public ArtifactConfigurationBuilder artifactLocation(File artifactLocation) {
    this.applicationLocation = artifactLocation;
    return this;
  }

  @Override
  public ArtifactConfigurationBuilder deploymentConfiguration(DeploymentConfiguration deploymentConfiguration) {
    this.deploymentConfiguration = deploymentConfiguration;
    return this;
  }

  @Override
  public ArtifactConfiguration build() {
    return new ArtifactConfigurationImpl(deploymentConfiguration, applicationLocation);
  }

  /**
   * Default implementation for {@link ArtifactConfiguration}
   */
  static class ArtifactConfigurationImpl implements ArtifactConfiguration {

    private static final long serialVersionUID = -4039186436472162679L;

    private DeploymentConfiguration deploymentConfiguration;
    private File applicationLocation;

    public ArtifactConfigurationImpl(DeploymentConfiguration deploymentConfiguration, File applicationLocation) {
      checkArgument(deploymentConfiguration != null, "deploymentConfiguration cannot be null");
      checkArgument(applicationLocation != null, "applicationLocation cannot be null");
      this.deploymentConfiguration = deploymentConfiguration;
      this.applicationLocation = applicationLocation;
    }

    @Override
    public DeploymentConfiguration getDeploymentConfiguration() {
      return deploymentConfiguration;
    }

    @Override
    public File getArtifactLocation() {
      return applicationLocation;
    }

  }

}
