/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.embedded.commons.api;

import org.mule.runtime.module.embedded.commons.internal.DefaultDeploymentConfigurationBuilder;

import java.io.Serializable;

/**
 * Specific deployment configuration for the artifact. This deployment configuration overrides default deployment configuration
 * for the artifact.
 *
 * @since 1.8, 1.7.2, 1.6.6
 */
public interface DeploymentConfiguration extends Serializable {

  /**
   * @return {@code true} if the artifact context will be created a lazily, {@code false} otherwise.
   */
  boolean lazyInitializationEnabled();

  /**
   * @return {@code true} if the artifact will be deployed with XSD validations, {@code false} otherwise.
   */
  boolean xmlValidationsEnabled();

  /**
   * @return {@code true} if the artifact will be deployed with the connections being created lazily meaning that the connection
   *         won't be established until an operation wants to use it, {@code false} otherwise.
   */
  boolean lazyConnectionsEnabled();

  /**
   * @return {@code false} if the Tooling objects will be created and made available in the registry, {@code true} otherwise.
   */
  boolean doNotAddToolingObjectsToRegistry();

  /**
   * @return {@code true} if the ArtifactAst for this deployment will be made available in the registry, {@code false} otherwise.
   */
  boolean addArtifactAstToRegistry();

  /**
   * @return a new builder for the {@link DeploymentConfiguration}
   */
  static DeploymentConfigurationBuilder builder() {
    return new DefaultDeploymentConfigurationBuilder();
  }

  /**
   * Builder interface for {@link DeploymentConfiguration}. Instances must be created using
   * {@link DeploymentConfiguration#builder()} method.
   */
  interface DeploymentConfigurationBuilder {

    /**
     * When an artifact context is lazily created then not all components will be created during the creation of the artifact.
     * Only the minimum set of required components will be created upon request of functionality. By default initialization is not
     * lazy.
     *
     * @return same builder
     */
    DeploymentConfigurationBuilder lazyInitialization(boolean enabled);

    /**
     * Allows to enable or disable XML validations while deploying the artifact. By default validations are enabled.
     *
     * @return same builder
     */
    DeploymentConfigurationBuilder xmlValidations(boolean enabled);

    /**
     * Allows to enable or disable if the artifact will be deployed with the connections being created lazily meaning that the
     * connection won't be established until an operation wants to use it. By default connections are created eagerly.
     * <p/>
     * If enabled, then connectivity testing will be always success.
     *
     * @return same builder
     */
    DeploymentConfigurationBuilder lazyConnectionsEnabled(boolean enabled);

    /**
     * Allows to configure whether the objects that support the tooling functionality will NOT be available in the registry.
     *
     * @return same builder
     */
    DeploymentConfigurationBuilder doNotAddToolingObjectsToRegistry(boolean enabled);

    /**
     * Allows to configure whether the ArtifactAst for the deployed artifact will be available in the registry.
     *
     * @return same builder
     */
    DeploymentConfigurationBuilder addArtifactAstToRegistry(boolean addArtifactAstToRegistry);

    /**
     * @return builds a {@link DeploymentConfiguration} deployment configuration instance with the provided configuration.
     */
    DeploymentConfiguration build();

  }

}
