/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.extension.internal;

import static java.lang.String.format;
import static java.util.Collections.emptyList;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toList;

import static javax.lang.model.element.ElementKind.METHOD;
import static net.bytebuddy.dynamic.scaffold.subclass.ConstructorStrategy.Default.IMITATE_SUPER_CLASS;
import static net.bytebuddy.implementation.MethodDelegation.to;
import static net.bytebuddy.matcher.ElementMatchers.any;
import static net.bytebuddy.matcher.ElementMatchers.isToString;

import org.mule.runtime.module.extension.api.loader.java.type.AnnotationValueFetcher;
import org.mule.runtime.module.extension.api.loader.java.type.Type;
import org.mule.sdk.api.meta.JavaVersion;

import java.lang.annotation.Annotation;
import java.lang.reflect.Array;
import java.lang.reflect.Method;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import java.util.function.Supplier;

import javax.annotation.processing.ProcessingEnvironment;
import javax.lang.model.element.AnnotationMirror;
import javax.lang.model.element.AnnotationValue;
import javax.lang.model.element.Element;
import javax.lang.model.element.ExecutableElement;
import javax.lang.model.element.TypeElement;
import javax.lang.model.element.VariableElement;
import javax.lang.model.type.DeclaredType;
import javax.lang.model.type.TypeMirror;

import net.bytebuddy.ByteBuddy;
import net.bytebuddy.implementation.FixedValue;
import net.bytebuddy.implementation.bind.annotation.AllArguments;
import net.bytebuddy.implementation.bind.annotation.Empty;
import net.bytebuddy.implementation.bind.annotation.Origin;
import net.bytebuddy.implementation.bind.annotation.RuntimeType;
import net.bytebuddy.implementation.bind.annotation.SuperMethod;
import net.bytebuddy.implementation.bind.annotation.This;

/**
 * {@link AnnotationValueFetcher} which works with the Java AST.
 *
 * @since 1.0
 */
public class ASTValueFetcher<A extends Annotation> implements AnnotationValueFetcher<A> {

  private Class<A> annotationClass;
  private final ProcessingEnvironment processingEnvironment;
  private final Supplier<Optional<AnnotationMirror>> annotationMirrorSupplier;

  ASTValueFetcher(Class<A> annotationClass, Element annotatedElememt, ProcessingEnvironment processingEnvironment) {
    this.annotationClass = annotationClass;
    this.processingEnvironment = processingEnvironment;
    this.annotationMirrorSupplier = () -> getAnnotationFrom(annotationClass, annotatedElememt, processingEnvironment);
  }

  private ASTValueFetcher(AnnotationMirror annotationMirror, ProcessingEnvironment processingEnvironment) {
    this.processingEnvironment = processingEnvironment;
    this.annotationMirrorSupplier = () -> ofNullable(annotationMirror);
    try {
      this.annotationClass = (Class<A>) Class.forName(annotationMirror.getAnnotationType().toString());
    } catch (ClassNotFoundException e) {
      throw new RuntimeException(format("Unable to create Annotation Value Fetcher for Annotation: [%s], it doesn't exist as a Class.",
                                        annotationMirror.getAnnotationType().toString()),
                                 e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getStringValue(Function<A, String> function) {
    return (String) getConstant(function).getValue();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public <E> List<E> getArrayValue(Function<A, E[]> function) {
    AnnotationValue annotationValue = (AnnotationValue) getObjectValue(function);
    if (annotationValue != null) {
      List<AnnotationValue> array = (List<AnnotationValue>) annotationValue.getValue();
      return array.stream().map(val -> (E) val.getValue()).collect(toList());
    } else {
      return emptyList();
    }
  }

  @Override
  public <E extends Enum> List<E> getEnumArrayValue(Function<A, E[]> function) {
    AnnotationValue annotationValue = (AnnotationValue) getObjectValue(function);
    if (annotationValue != null) {
      List<AnnotationValue> array = (List<AnnotationValue>) annotationValue.getValue();

      if (!array.isEmpty()) {
        Class<? extends Enum> enumAnnotationClass;
        try {
          VariableElement firstItemValue = (VariableElement) array.get(0).getValue();
          enumAnnotationClass = (Class<? extends Enum>) Class
              .forName(processingEnvironment.getElementUtils().getBinaryName((TypeElement) firstItemValue.getEnclosingElement())
                  .toString());
        } catch (ClassNotFoundException e) {
          throw new RuntimeException(e);
        }
        return array.stream().map(val -> {
          try {
            return (E) getEnumValue(val, enumAnnotationClass);
          } catch (IllegalArgumentException e) {
            // There might be `JavaVersion` values that are present in the extension's SDK API version but not in the packager's
            // one. The ones that are available should be preserved
            if (enumAnnotationClass.equals(JavaVersion.class)) {
              return null;
            }

            throw e;
          }
        })
            .filter(Objects::nonNull)
            .collect(toList());
      }
    }
    return emptyList();
  }

  protected final <E extends Enum> E getEnumValue(AnnotationValue val, Class<? extends Enum> enumAnnotationClass) {
    return (E) Enum.valueOf(enumAnnotationClass, val.getValue().toString());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<Type> getClassArrayValue(Function<A, Class[]> function) {
    AnnotationValue value = (AnnotationValue) getObjectValue(function);
    if (value != null) {
      List<AnnotationValue> array = (List<AnnotationValue>) value.getValue();
      return array.stream().map(attr -> ((DeclaredType) attr.getValue()))
          .map(declaredType -> new ASTType((TypeElement) declaredType.asElement(), processingEnvironment))
          .collect(toList());
    } else {
      return emptyList();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ASTType getClassValue(Function<A, Class> function) {
    Object objectValue = getObjectValue(function);
    if (objectValue == null) {
      return null;
    }
    return new ASTType((TypeElement) ((DeclaredType) ((AnnotationValue) objectValue).getValue()).asElement(),
                       processingEnvironment);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public <N extends Number> N getNumberValue(Function<A, N> function) {
    return (N) getConstant(function).getValue();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Boolean getBooleanValue(Function<A, Boolean> function) {
    return (Boolean) getConstant(function).getValue();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public <E extends Enum> E getEnumValue(Function<A, E> function) {
    VariableElement value = (VariableElement) ((AnnotationValue) getObjectValue(function)).getValue();
    Class<? extends Enum> enumAnnotationClass;
    try {
      enumAnnotationClass = (Class<? extends Enum>) Class
          .forName(processingEnvironment.getElementUtils().getBinaryName((TypeElement) value.getEnclosingElement()).toString());
    } catch (ClassNotFoundException e) {
      throw new RuntimeException(e);
    }
    return (E) Enum.valueOf(enumAnnotationClass, value.toString());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public <E extends Annotation> AnnotationValueFetcher<E> getInnerAnnotation(Function<A, E> function) {
    AnnotationValue annotationValue = (AnnotationValue) getObjectValue(function);
    return new ASTValueFetcher<>((AnnotationMirror) annotationValue.getValue(), processingEnvironment);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public <E extends Annotation> List<AnnotationValueFetcher<E>> getInnerAnnotations(Function<A, E[]> function) {
    List<AnnotationMirror> annotationMirrors = (List<AnnotationMirror>) ((AnnotationValue) getObjectValue(function)).getValue();
    return annotationMirrors.stream()
        .map(am -> new ASTValueFetcher<>(am, processingEnvironment))
        .map(fetcher -> (AnnotationValueFetcher<E>) fetcher)
        .collect(toList());
  }

  private AnnotationValue getConstant(Function function) {
    return (AnnotationValue) getObjectValue(function);
  }

  private Object getObjectValue(Function function) {
    return getObjectValue(annotationClass, function, annotationMirrorSupplier);
  }

  /**
   * Retrieves the value of an annotation property.
   * <p>
   * The problem with Annotations in a AST environment is that the way to obtain Annotation values behaves different between
   * having a proper Annotation class and one when using the AST, because for example if there is an annotation that has Class
   * references, you will never be able to obtain the Class value, because that class doesn't exist already As consequence the
   * safe way to access annotation values in AST is knowing the property name, this mean soft references.
   * <p>
   * To prevent the soft reference, the user obtains a Function which receives an Annotation instance and later makes usage of the
   * annotation to communicate which property wants to retrieve.
   * <p>
   * E.g.: {@code getAnnotationValue(MetadataKeyPart.class).intValue(MetadataKeyPart::order);}
   * {@code getAnnotationValue(OfValues.class).classValue(OfValues::value);}
   * <p>
   * But this doesn't fix the issue, because asking for the class in a AST environment will make everything fail, the key part is
   * that the Annotation instance is not a real instance, is a proxy which intercepts the call with the purpose of obtain the
   * property names this method can safely return the property value.
   *
   * @param annotationClass        The annotation to look for
   * @param retrievalValueFunction Function to obtain the property name
   * @param annotationMirror       Returns the annotation object to introspect
   * @param <T>                    The annotation type
   * @return The annotation property value
   */
  private <T> Object getObjectValue(Class<T> annotationClass, Function<T, Object> retrievalValueFunction,
                                    Supplier<Optional<AnnotationMirror>> annotationMirror) {

    ReferenceInterceptor refInterceptor = new ReferenceInterceptor(annotationMirror);
    Class<? extends T> annotatedClass = new ByteBuddy().subclass(annotationClass, IMITATE_SUPER_CLASS)
        // This is just to prevent to be called with a toString() by the IDE when debugging
        .method(isToString()).intercept(FixedValue.value("string"))
        .method(any()).intercept(to(refInterceptor))
        .make().load(annotationClass.getClassLoader()).getLoaded();
    try {
      T obj = annotatedClass.getConstructor().newInstance();
      retrievalValueFunction.apply(obj);
    } catch (Exception e) {
      throw new IllegalStateException("Could not create instance of annotated version of " + annotationClass.getName(), e);
    }
    return refInterceptor.getReference();
  }

  private static Optional<AnnotationMirror> getAnnotationFrom(Class<?> configurationClass, Element typeElement,
                                                              ProcessingEnvironment processingEnvironment) {
    TypeElement annotationTypeElement = processingEnvironment.getElementUtils().getTypeElement(configurationClass.getTypeName());
    for (AnnotationMirror annotationMirror : typeElement.getAnnotationMirrors()) {
      DeclaredType annotationType = annotationMirror.getAnnotationType();
      TypeMirror obj = annotationTypeElement.asType();

      if (processingEnvironment.getTypeUtils().isSameType(annotationType, obj)) {
        return of(annotationMirror);
      }
    }
    return empty();
  }

  public static class ReferenceInterceptor {

    private Object reference = null;
    private final Supplier<Optional<AnnotationMirror>> annotationMirror;


    public ReferenceInterceptor(Supplier<Optional<AnnotationMirror>> annotationMirror) {
      this.annotationMirror = annotationMirror;
    }

    @RuntimeType
    public Object intercept(@This Object object, @Origin Method method, @AllArguments Object[] args,
                            @SuperMethod(nullIfImpossible = true) Method superMethod, @Empty Object defaultValue) {
      this.reference = null;

      annotationMirror.get()
          .ifPresent(annotation -> getAnnotationElementValue(annotation, method.getName()).ifPresent(this::setReference));
      // We just return the default type of the return type. We can't just return null because then we will get a NPE
      // when dealing with native types.
      return getDefaultValue(method.getReturnType());
    }

    private void setReference(Object object) {
      this.reference = object;
    }

    public Object getReference() {
      return reference;
    }
  }

  private static Optional<? extends AnnotationValue> getAnnotationElementValue(AnnotationMirror annotation, String name) {
    for (Map.Entry<? extends ExecutableElement, ? extends AnnotationValue> entry : annotation.getElementValues().entrySet()) {
      if (entry.getKey().getSimpleName().toString().equals(name)) {
        return of(entry.getValue());
      }
    }

    for (Element element : annotation.getAnnotationType().asElement().getEnclosedElements()) {
      if (element.getKind().equals(METHOD) && element.getSimpleName().toString().equals(name)) {
        return ofNullable(((ExecutableElement) element).getDefaultValue());
      }
    }

    return empty();
  }

  private static <T> T getDefaultValue(Class<T> clazz) {
    return (T) Array.get(Array.newInstance(clazz, 1), 0);
  }

}
