/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.extension.internal;

import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.runtime.parameter.Literal;
import org.mule.runtime.extension.api.runtime.parameter.ParameterResolver;

import javax.annotation.processing.ProcessingEnvironment;
import javax.lang.model.type.TypeMirror;
import javax.lang.model.util.Types;

/**
 * A helper class to handle types with the AST
 *
 * @since 1.4
 */
public final class ExtensionTypeUtils {

  private ExtensionTypeUtils() {}

  public static boolean isParameterResolverType(TypeMirror typeMirror, ProcessingEnvironment processingEnvironment) {
    return isSameType(typeMirror, ParameterResolver.class, processingEnvironment)
        || isSameType(typeMirror, org.mule.sdk.api.runtime.parameter.ParameterResolver.class, processingEnvironment);
  }

  public static boolean isLiteralType(TypeMirror typeMirror, ProcessingEnvironment processingEnvironment) {
    return isSameType(typeMirror, Literal.class, processingEnvironment)
        || isSameType(typeMirror, org.mule.sdk.api.runtime.parameter.Literal.class, processingEnvironment);
  }

  public static boolean isTypedValueType(TypeMirror typeMirror, ProcessingEnvironment processingEnvironment) {
    return isSameType(typeMirror, TypedValue.class, processingEnvironment);
  }


  private static boolean isSameType(TypeMirror typeMirror, Class<?> aClass, ProcessingEnvironment processingEnvironment) {
    Types types = processingEnvironment.getTypeUtils();
    TypeMirror erasure = types.erasure(typeMirror);
    TypeMirror clazz = types.erasure(processingEnvironment.getElementUtils().getTypeElement(aClass.getName()).asType());

    return types.isSameType(erasure, clazz);
  }
}
