/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Stream.concat;

import org.mule.runtime.extension.api.annotation.ExpressionFunctions;
import org.mule.runtime.extension.api.annotation.Operations;
import org.mule.runtime.extension.api.annotation.Sources;
import org.mule.runtime.extension.api.annotation.connectivity.ConnectionProviders;
import org.mule.runtime.module.extension.api.loader.java.type.ConfigurationElement;
import org.mule.runtime.module.extension.api.loader.java.type.ConnectionProviderElement;
import org.mule.runtime.module.extension.api.loader.java.type.FunctionContainerElement;
import org.mule.runtime.module.extension.api.loader.java.type.OperationContainerElement;
import org.mule.runtime.module.extension.api.loader.java.type.SourceElement;
import org.mule.runtime.module.extension.api.loader.java.type.Type;

import java.lang.annotation.Annotation;
import java.util.List;
import java.util.function.Function;
import java.util.stream.Stream;

import javax.annotation.processing.ProcessingEnvironment;
import javax.lang.model.element.TypeElement;

/**
 * {@link ConfigurationElement}
 *
 * @since 1.0
 */
public class ConfigurationASTElement extends ASTType implements ConfigurationElement {

  public ConfigurationASTElement(TypeElement typeElement, ProcessingEnvironment processingEnvironment) {
    super(typeElement, processingEnvironment);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<ConnectionProviderElement> getConnectionProviders() {
    return concat(
                  collectConnectionProviders(ConnectionProviders.class, ConnectionProviders::value),
                  collectConnectionProviders(org.mule.sdk.api.annotation.connectivity.ConnectionProviders.class,
                                             org.mule.sdk.api.annotation.connectivity.ConnectionProviders::value))
                                                 .collect(toList());
  }

  private <A extends Annotation> Stream<ConnectionProviderElement> collectConnectionProviders(Class<A> annotationType,
                                                                                              Function<A, Class[]> extractionFunction) {
    return collectElements(
                           annotationType,
                           extractionFunction,
                           element -> new ConnectionProviderASTElement((element).getElement().get(), processingEnvironment));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<FunctionContainerElement> getFunctionContainers() {
    return getFunctionContainerStream().collect(toList());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<SourceElement> getSources() {
    return getSourceElementStream().collect(toList());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<OperationContainerElement> getOperationContainers() {
    return getOperationElementStream().collect(toList());
  }

  protected Stream<FunctionContainerElement> getFunctionContainerStream() {
    return concat(
                  collectFunctionElements(ExpressionFunctions.class, ExpressionFunctions::value),
                  collectFunctionElements(
                                          org.mule.sdk.api.annotation.ExpressionFunctions.class,
                                          org.mule.sdk.api.annotation.ExpressionFunctions::value));
  }

  protected Stream<SourceElement> getSourceElementStream() {
    return concat(
                  collectSourceElements(Sources.class, Sources::value),
                  collectSourceElements(org.mule.sdk.api.annotation.Sources.class, org.mule.sdk.api.annotation.Sources::value));
  }

  protected Stream<OperationContainerElement> getOperationElementStream() {
    return concat(
                  collectOperationElements(Operations.class, Operations::value),
                  collectOperationElements(org.mule.sdk.api.annotation.Operations.class,
                                           org.mule.sdk.api.annotation.Operations::value));
  }

  protected <A extends Annotation> Stream<SourceElement> collectSourceElements(
                                                                               Class<A> annotationClass,
                                                                               Function<A, Class[]> extractionFunction) {
    return collectElements(annotationClass, extractionFunction,
                           type -> new SourceElementAST(type.getElement().get(), processingEnvironment));
  }

  protected <A extends Annotation> Stream<OperationContainerElement> collectOperationElements(
                                                                                              Class<A> annotationClass,
                                                                                              Function<A, Class[]> extractionFunction) {
    return collectElements(annotationClass, extractionFunction,
                           type -> new OperationContainerElementAST(type.getElement().get(), processingEnvironment));
  }

  protected <A extends Annotation> Stream<FunctionContainerElement> collectFunctionElements(
                                                                                            Class<A> annotationClass,
                                                                                            Function<A, Class[]> extractionFunction) {
    return collectElements(annotationClass, extractionFunction,
                           type -> new FunctionContainerASTElement(type.getElement().get(), processingEnvironment));
  }

  protected <A extends Annotation, T> Stream<T> collectElements(Class<A> annotationClass,
                                                                Function<A, Class[]> extractionFunction,
                                                                Function<Type, T> elementFunction) {
    return getValueFromAnnotation(annotationClass)
        .map(valueFetcher -> valueFetcher.getClassArrayValue(extractionFunction)
            .stream()
            .map(elementFunction))
        .orElse(Stream.empty());
  }
}
