/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal;

import static java.util.stream.Collectors.toList;
import static org.mule.runtime.extension.internal.loader.util.JavaParserUtils.toMuleApi;

import org.mule.runtime.api.functional.Either;
import org.mule.runtime.api.meta.Category;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.extension.api.annotation.Configurations;
import org.mule.runtime.extension.api.annotation.Extension;
import org.mule.runtime.extension.api.exception.IllegalModelDefinitionException;
import org.mule.runtime.module.extension.api.loader.java.type.AnnotationValueFetcher;
import org.mule.runtime.module.extension.api.loader.java.type.ConfigurationElement;
import org.mule.runtime.module.extension.api.loader.java.type.ExtensionElement;
import org.mule.runtime.module.extension.api.loader.java.type.FunctionContainerElement;
import org.mule.runtime.module.extension.api.loader.java.type.FunctionElement;
import org.mule.runtime.module.extension.api.loader.java.type.OperationContainerElement;
import org.mule.runtime.module.extension.api.loader.java.type.OperationElement;

import java.lang.annotation.Annotation;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Stream;

import javax.annotation.processing.ProcessingEnvironment;
import javax.lang.model.element.TypeElement;

/**
 * {@link ExtensionElement} implementation which works with the Java AST.
 *
 * @since 1.0
 */
public class ExtensionTypeElement extends ConfigurationASTElement implements ExtensionElement {

  private LazyValue<AnnotationReader> extensionAnnotation = new LazyValue<>(AnnotationReader::new);

  /**
   * @param typeElement
   * @param processingEnv
   */
  public ExtensionTypeElement(TypeElement typeElement, ProcessingEnvironment processingEnv) {
    super(typeElement, processingEnv);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<ConfigurationElement> getConfigurations() {
    return Stream.concat(
                         fetchConfigurationElements(Configurations.class, Configurations::value),
                         fetchConfigurationElements(org.mule.sdk.api.annotation.Configurations.class,
                                                    org.mule.sdk.api.annotation.Configurations::value))
        .collect(toList());
  }

  private <T extends Annotation> Stream<ConfigurationElement> fetchConfigurationElements(Class<T> annotationClass,
                                                                                         Function<T, Class[]> mapper) {
    return getValueFromAnnotation(annotationClass)
        .map(valFetcher -> valFetcher
            .getClassArrayValue(mapper)
            .stream()
            .map(configType -> (ConfigurationElement) new ConfigurationASTElement(configType.getElement().get(),
                                                                                  processingEnvironment)))
        .orElse(Stream.empty());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<FunctionElement> getFunctions() {
    return getFunctionContainerStream()
        .map(FunctionContainerElement::getFunctions)
        .flatMap(Collection::stream)
        .collect(toList());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<OperationElement> getOperations() {
    return getOperationElementStream()
        .map(OperationContainerElement::getOperations)
        .flatMap(Collection::stream)
        .collect(toList());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Category getCategory() {
    return extensionAnnotation.get().getCategory();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getVendor() {
    return extensionAnnotation.get().getVendor();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getName() {
    return extensionAnnotation.get().getName();
  }

  private class AnnotationReader {

    private final Either<AnnotationValueFetcher<Extension>, AnnotationValueFetcher<org.mule.sdk.api.annotation.Extension>> either;

    private AnnotationReader() {
      Optional<AnnotationValueFetcher<Extension>> left = getValueFromAnnotation(Extension.class);
      if (left.isPresent()) {
        either = Either.left(left.get());
      } else {
        Optional<AnnotationValueFetcher<org.mule.sdk.api.annotation.Extension>> right =
            getValueFromAnnotation(org.mule.sdk.api.annotation.Extension.class);
        if (right.isPresent()) {
          either = Either.right(right.get());
        } else {
          throw new IllegalModelDefinitionException(String.format("Extension is not annotated with neither '%s' nor '%s'",
                                                                  Extension.class.getName(),
                                                                  org.mule.sdk.api.annotation.Extension.class.getName()));
        }
      }
    }

    private String getName() {
      return either.reduce(
                           v -> v.getStringValue(Extension::name),
                           v -> v.getStringValue(org.mule.sdk.api.annotation.Extension::name));
    }

    private String getVendor() {
      return either.reduce(
                           v -> v.getStringValue(Extension::vendor),
                           v -> v.getStringValue(org.mule.sdk.api.annotation.Extension::vendor));
    }

    private Category getCategory() {
      return either.reduce(
                           v -> v.getEnumValue(Extension::category),
                           v -> toMuleApi(v.getEnumValue(org.mule.sdk.api.annotation.Extension::category)));
    }
  }
}
