/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.features.internal.generator;

import static java.nio.file.Files.createDirectories;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Set;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;

import static java.lang.reflect.Modifier.isFinal;
import static java.lang.reflect.Modifier.isPublic;
import static java.lang.reflect.Modifier.isStatic;

/**
 * Abstract base class for generating Java source code files during the build process. This class provides common functionality
 * for creating Java files with proper package structure, imports, and content generation.
 *
 * <p>
 * Subclasses must implement the abstract methods to define the specific content and structure of the generated Java class.
 * </p>
 */
abstract class AbstractClassGenerator {

  private final File outputDir;

  /**
   * Constructs a new AbstractClassGenerator with the specified output directory.
   *
   * @param outputDir the directory where the generated Java files will be written
   */
  AbstractClassGenerator(File outputDir) {
    this.outputDir = outputDir;
  }

  /**
   * Returns the package name for the generated Java class.
   *
   * @return the fully qualified package name
   */
  protected abstract String getPackageName();

  /**
   * Returns the set of import statements needed for the generated Java class.
   *
   * @return a set of fully qualified class names to import
   */
  protected abstract Set<String> getImports();

  /**
   * Returns the simple class name for the generated Java class.
   *
   * @return the class name without package prefix
   */
  protected abstract String getClassName();

  /**
   * Writes the main content of the Java class to the output stream. This method should write the class declaration, fields,
   * methods, and any other class-specific content.
   *
   * @param outputStream the output stream to write the class content to
   * @throws IOException if an I/O error occurs while writing
   */
  protected abstract void writeClassContent(OutputStream outputStream) throws IOException;

  /**
   * Appends a line of text followed by a newline character to the output stream.
   *
   * @param outputStream the output stream to write to
   * @param line         the text line to append
   * @throws IOException if an I/O error occurs while writing
   */
  protected void appendLine(OutputStream outputStream, String line) throws IOException {
    outputStream.write((line + "\n").getBytes());
  }

  /**
   * Appends an empty line (newline character) to the output stream.
   *
   * @param outputStream the output stream to write to
   * @throws IOException if an I/O error occurs while writing
   */
  protected void appendLine(OutputStream outputStream) throws IOException {
    appendLine(outputStream, "");
  }

  /**
   * Generates the Java source file by creating the appropriate directory structure, writing the package declaration, import
   * statements, and class content.
   *
   * @throws IOException if an I/O error occurs during file creation or writing
   */
  public void generate() throws IOException {
    Path packageLocation = Paths.get(outputDir.getAbsolutePath(), getLocationFromPackageName(getPackageName()));
    createDirectories(packageLocation);
    File generatedJava = packageLocation.resolve(getClassName() + ".java").toFile();
    generatedJava.createNewFile();
    try (FileOutputStream outputStream = new FileOutputStream(generatedJava)) {
      appendLine(outputStream, "package " + getPackageName() + ";");
      appendLine(outputStream);

      for (String importedClass : getImports()) {
        appendLine(outputStream, "import " + importedClass + ";");
      }

      writeClassContent(outputStream);
    }
  }

  private static String getLocationFromPackageName(String packageName) {
    return packageName.replace(".", "/");
  }

  /**
   * Checks if a field is a public static final String constant.
   *
   * @param field the field to check
   * @return true if the field is public, static, final, and of type String
   */
  protected static boolean isPublicStaticFinalString(Field field) {
    int modifiers = field.getModifiers();
    return isPublic(modifiers) && isStatic(modifiers) && isFinal(modifiers) && field.getType().equals(String.class);
  }

  /**
   * Checks if a field is a public static final field of the specified feature class type.
   *
   * @param field        the field to check
   * @param featureClass the expected type of the field
   * @return true if the field is public, static, final, and of the specified feature class type
   */
  protected static boolean isPublicStaticFinalFeature(Field field, Class<?> featureClass) {
    int modifiers = field.getModifiers();
    return isPublic(modifiers) && isStatic(modifiers) && isFinal(modifiers)
        && field.getType().equals(featureClass);
  }

  /**
   * Checks if an annotation class is available and not from the legacy Mule API annotation package.
   *
   * @param annotation the annotation class to check
   * @return true if the annotation is available and not from the org.mule.api.annotation package
   */
  protected static boolean isAvailableAnnotation(Class<? extends Annotation> annotation) {
    return !annotation.getPackageName().startsWith("org.mule.api.annotation");
  }

  /**
   * Checks if a class needs to be imported
   *
   * @param cls the class to check
   * @return true if the class needs an import statement
   */
  protected static boolean isImportNeeded(Class<?> cls) {
    return !cls.getPackageName().startsWith("java.lang");
  }
}
