/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.http.api.domain.entity;

import static org.mule.runtime.api.util.MultiMap.emptyMultiMap;

import static java.util.Objects.requireNonNull;
import static java.util.OptionalLong.empty;
import static java.util.OptionalLong.of;
import static java.util.concurrent.CompletableFuture.completedFuture;

import org.mule.api.annotation.NoExtend;
import org.mule.runtime.api.util.MultiMap;

import java.io.IOException;
import java.io.InputStream;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.concurrent.CompletableFuture;
import java.util.function.BiConsumer;

import org.apache.commons.io.IOUtils;

/**
 * Representation of a stream HTTP body.
 *
 * @since 4.0
 */
@NoExtend
public class InputStreamHttpEntity implements HttpEntity {

  private OptionalLong contentLength;
  private InputStream inputStream;
  private CompletableFuture<MultiMap<String, String>> futureTrailers;

  public InputStreamHttpEntity(InputStream inputStream) {
    requireNonNull(inputStream, "HTTP entity stream cannot be null.");
    this.inputStream = inputStream;
    this.contentLength = empty();
    this.futureTrailers = completedFuture(emptyMultiMap());
  }

  /**
   * @deprecated Use {@link #InputStreamHttpEntity(InputStream, OptionalLong)} or
   *             {@link #InputStreamHttpEntity(InputStream, long)} instead.
   */
  @Deprecated
  public InputStreamHttpEntity(InputStream inputStream, Long contentLength) {
    this(inputStream);
    this.contentLength = contentLength == null ? empty() : of(contentLength);
  }

  /**
   * @since 4.2
   */
  public InputStreamHttpEntity(InputStream inputStream, long contentLength) {
    this(inputStream);
    this.contentLength = of(contentLength);
  }

  /**
   * @since 4.2
   */
  public InputStreamHttpEntity(InputStream inputStream, OptionalLong contentLength) {
    this(inputStream);
    this.contentLength = contentLength;
  }

  public InputStreamHttpEntity(InputStream stream, long contentLengthAsLong,
                               CompletableFuture<MultiMap<String, String>> futureTrailers) {
    this(stream, contentLengthAsLong);
    this.futureTrailers = futureTrailers;
  }

  public InputStreamHttpEntity(InputStream stream, CompletableFuture<MultiMap<String, String>> futureTrailers) {
    this(stream);
    this.futureTrailers = futureTrailers;
  }

  @Override
  public boolean isStreaming() {
    return true;
  }

  @Override
  public InputStream getContent() {
    return this.inputStream;
  }

  @Override
  public byte[] getBytes() throws IOException {
    return IOUtils.toByteArray(this.inputStream);
  }

  @Override
  public Optional<Long> getLength() {
    return contentLength.isPresent() ? Optional.of(contentLength.getAsLong()) : Optional.empty();
  }

  @Override
  public OptionalLong getBytesLength() {
    return contentLength;
  }

  @Override
  public void onComplete(BiConsumer<? super MultiMap<String, String>, ? super Throwable> completionCallback) {
    futureTrailers.whenComplete(completionCallback);
  }
}
